# -*- coding: utf-8 -*-
import logging

from django.contrib import admin
from django.contrib.admin import SimpleListFilter
from django.contrib import messages
from django.contrib.auth.models import User
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from django.utils.translation import ungettext
from django.shortcuts import get_object_or_404

from mlcore.ml.forms import MailListField, BulkUserSubscribeForm
from mlcore.permissions.utils import is_external_staff
from mlcore.subscribe.utils import find_available_backends, load_backend_interface
from mlcore.ml.models import (MailList, Subscribers, Owner, EmailSubscriber, LoginRedirect,
                              Autosubscription, OperationLog, CreateListLock, CorporateDomain)

from mlcore.ml.operation_choices import OperationChoices
from mlcore.tasks import retry_operations
from mlcore.tasks import operations
from mlcore.tasks.operations import (subscribe_user, subscribe_email, unsubscribe,
                                     resubscribe)
from mlcore.utils.getters import get_user

from django_intranet_stuff.forms.fields import StaffUserField, GroupField
from django_intranet_stuff.models import Staff

from django.conf.urls import patterns, url, include
from django.http import HttpResponse

import class_views


logger = logging.getLogger(__name__)


backend_context_inlines = []
for backend_type in find_available_backends():
    backend = load_backend_interface(backend_type)

    class Inline(admin.TabularInline):
        model = backend.model_class
        extra = 0
        exclude = ('backend_type',)
    backend_context_inlines.append(Inline)


class MaillistAdmin(admin.ModelAdmin):

    list_display = ("name", "email", "is_open", "is_deleted", "relevance",
                    "created_at")
    list_filter = ("letter", "is_imap", "is_sub", "is_open", "is_deleted")
    ordering = ("name",)
    search_fields = ("name", "email")
    inlines = backend_context_inlines
    exclude = ("created_at", "created_dt", "sub_users", "imap_users")
    actions = ('mark_deleted', 'unmark_deleted')

    def mark_deleted(self, request, queryset):
        nupdated = queryset.update(is_deleted=True)
        # Обновляем кэш NWSMTP ML-1555
        for maillist in queryset:
            maillist.set_as_modified()
        messages.info(request,
                ungettext(
                    "Marked %(nupdated)d maillist as deleted",
                    "Marked %(nupdated)d maillists as deleted",
                    nupdated) % {'nupdated': nupdated})
    mark_deleted.short_description = _("Mark maillists as deleted")

    def unmark_deleted(self, request, queryset):
        nupdated = queryset.update(is_deleted=False)
        # Обновляем кэш NWSMTP ML-1555
        for maillist in queryset:
            maillist.set_as_modified()
        messages.info(request,
                ungettext(
                    "Unmarked deletion in %(nupdated)s maillist",
                    "Unmarked deletion in %(nupdated)s maillists",
                    nupdated) % {'nupdated': nupdated})
    unmark_deleted.short_description = _("Unmark maillists as deleted")

admin.site.register(MailList, MaillistAdmin)


class SubscriberAdmin(admin.ModelAdmin):

    class Media:
        js = ('https://yandex.st/jquery/1.4.2/jquery.min.js',)

    list_display = ("list", "user", "is_imap", "is_sub")
    list_filter = ("is_imap", "is_sub")
    ordering = ("user__username",)
    search_fields = ("user__username", "user__email", "list__name", "=list__email")
    actions = ('force_unsubscribe', 'force_resubscribe')

    add_form_template = 'admin/ml/add_subscribers.html'

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == 'user':
            return StaffUserField()
        if db_field.name == 'list':
            return MailListField()
        return super(SubscriberAdmin, self)\
                .formfield_for_foreignkey(db_field, request, **kwargs)

    def force_resubscribe(self, request, queryset):
        for sub in queryset.select_related():
            resubscribe.delay({'initiator': request.user.username,
                               'comment': u'Переподписка'},
                              user=sub.user, maillist=sub.list)
        messages.info(request, u"Выбранные пользователи будут переподписаны.")

    force_resubscribe.short_description = u"Переподписать"

    def force_unsubscribe(self, request, queryset):
        for sub in queryset.select_related():
            unsubscribe.delay({'initiator': request.user.username,
                               'comment': u'Массовая отписка через админку'},
                              user=sub.user, maillist=sub.list)
        messages.info(request, u"Выбранные пользователи будут отписаны.")

    force_unsubscribe.short_description = _("Unsubscribe user from maillist (subscription stays in db)")

    def add_view(self, request, form_url='', extra_context=None):
        context = {'subscribe_form': BulkUserSubscribeForm(), 'is_popup': False,
                   'show_delete': False}
        if not request.POST:
            context.update(extra_context or {})
            return self.render_change_form(request, context, form_url=form_url,
                                           add=True)
        else:
            f = BulkUserSubscribeForm(request.POST)
            context['subscribe_form'] = f
            context.update(extra_context or {})

        if f.is_valid():
            maillist = get_object_or_404(MailList, name=f.cleaned_data['name'])
            subs_type = f.cleaned_data['type']
            logins_and_mails = f.cleaned_data['logins_and_mails']

            mails = set()
            logins = set()

            for item in logins_and_mails:
                if '@' in item:
                    try:
                        logins.add(get_user(item).username)
                    except User.DoesNotExist:
                        mails.add(item)
                else:
                    logins.add(item)

            ignore = set(Subscribers.objects.filter(
                user__username__in=logins, list=maillist).values_list(
                    'user__username', flat=True))

            without_email = set(Staff.objects.filter(
                login__in=logins, work_email=None).values_list('login', flat=True))

            # оставляем только тех кого можем подписать
            bads = set()
            external_ignored = set()
            for login in logins - (ignore | without_email):
                try:
                    user = User.objects.get(username=login)
                except User.DoesNotExist:
                    bads.add(login)
                    continue
                else:
                    if not maillist.external_staff_can_read and is_external_staff(user):
                        external_ignored.add(user.username)
                    else:
                        subscribe_user.delay(
                            {'initiator': request.user.username,
                             'comment': u'Массовая подписка через админку'},
                            user=user, maillist=maillist, type=subs_type,
                            check_rights=False)

            for email in mails:
                subscribe_email.delay(
                    {'initiator': request.user.username,
                     'comment': u'Массовая подписка через админку'},
                    email=email, maillist=maillist)

            if ignore:
                messages.warning(request, u"Уже подписаны на %s@: %s" %
                                 (maillist.name, ', '.join(ignore)))
            if without_email:
                messages.error(request, u"Небыли подписаны, из-за отсутствия" +
                               u" у них емейла: %s" % ', '.join(without_email))
            if bads:
                messages.error(request, u"Несуществующие логины: %s"
                               % ', '.join(bads))

            if external_ignored:
                messages.error(request, u"Небыли подписаны, т.к. являются внешними сотрудниками: %s" %
                               ', '.join(external_ignored))

            all_subscribed = (logins - (ignore | without_email | bads | external_ignored)) | mails
            if all_subscribed:
                messages.success(request, u"Будут подписаны на %s@: %s" %
                                 (maillist.name, ', '.join(all_subscribed)))
        return self.render_change_form(request, context, form_url=form_url,
                                       add=True)

admin.site.register(Subscribers, SubscriberAdmin)


class OwnerAdmin(admin.ModelAdmin):

    class Media:
        js = ('https://yandex.st/jquery/1.4.2/jquery.min.js',)

    ordering = ("user__username",)
    search_fields = ("user__username", "user__email", "list__name", "list__email")

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == 'user':
            return StaffUserField()
        if db_field.name == 'list':
            return MailListField()
        return super(OwnerAdmin, self)\
                .formfield_for_foreignkey(db_field, request, **kwargs)

admin.site.register(Owner, OwnerAdmin)


class EmailSubscriberAdmin(admin.ModelAdmin):
    ordering = ("email",)
    search_fields = ("email", "list__name", "=list__email")
    actions = ('unsubscribe', 'subscribe')

    def unsubscribe(self, request, queryset):
        import mlcore.subscribe.email
        for email_sub in queryset.select_related():
            mlcore.subscribe.email.unsubscribe(email_sub.list, email_sub.email)

    unsubscribe.short_description = _("Unsubscribe selected email subscriptions (subscriptions stays in db)")

    def subscribe(self, request, queryset):
        import mlcore.subscribe.email
        for email_sub in queryset.select_related():
            mlcore.subscribe.email.subscribe(email_sub.list, email_sub.email)

    subscribe.short_description = _("Subscribe selected emails in backends")


admin.site.register(EmailSubscriber, EmailSubscriberAdmin)


class AutosubscriptionAdmin(admin.ModelAdmin):
    list_display = ("group", "maillist",)

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == 'maillist':
            return MailListField()
        if db_field.name == 'group':
            return GroupField()
        return super(AutosubscriptionAdmin, self)\
                    .formfield_for_foreignkey(db_field, request, **kwargs)

    class Media:
        js = (
            'https://yandex.st/jquery/1.4.2/jquery.min.js',
            'https://%s/media/js/_js-autocomplete_groups.js' %
                                                    settings.CENTER_MASTER_WWW,
        )


admin.site.register(Autosubscription, AutosubscriptionAdmin)


class RootFilter(SimpleListFilter):
    title = u'Root'
    parameter_name = 'is_root'

    def lookups(self, request, model_admin):
        return (('root', u'Начальная операция'),)

    def queryset(self, request, queryset):
        if self.value() == 'root':
            return queryset.filter(parent=None)


class StatusFilter(SimpleListFilter):
    title = u'Статус'
    parameter_name = 'status'

    def lookups(self, request, model_admin):
        return (('FAILURE', u'Сломанные'),
                ('RETRIED', u'С перезапусками'),
                ('PENDING', u'В работе'))

    def queryset(self, request, queryset):
        if self.value() in ('FAILURE', 'RETRIED', 'PENDING', 'OBSOLETE'):
            logs = OperationLog.objects.filter(
                status=self.value(),
                root__in=queryset).values_list('root', flat=True)
            return queryset.filter(task_id__in=logs)


class ObsoleteFilter(StatusFilter):
    title = u'Устаревшие'
    parameter_name = 'obsolete'

    def lookups(self, request, model_admin):
        return (('NOT_OBSOLETE', u'Без устаревших'),
                ('OBSOLETE', u'Устаревшие'))

    def queryset(self, request, queryset):
        if self.value() == 'NOT_OBSOLETE':
            logs = OperationLog.objects.filter(
                status='OBSOLETE',
                root__in=queryset).values_list('root', flat=True)
            return queryset.exclude(task_id__in=logs)
        elif self.value() == 'OBSOLETE':
            return super(ObsoleteFilter, self).queryset(request, queryset)


class InitiatorFilter(SimpleListFilter):
    title = u'Инициатор'
    parameter_name = 'roboinitiator'

    def lookups(self, request, model_admin):
        return (('robot', u'Робот'),
                ('user', u'Человек'))

    def queryset(self, request, queryset):
        if self.value() in ('robot', 'user'):
            users = (User.objects
                     .filter(username__in=queryset.values_list('initiator'))
                     .values_list('username'))

        if self.value() == 'robot':
            return queryset.exclude(initiator__in=users)
        elif self.value() == 'user':
            return queryset.filter(initiator__in=users)


class OperationFilter(SimpleListFilter):
    title = u'Операция'
    parameter_name = 'name'
    CHOICES = OperationChoices(operations)

    def lookups(self, request, model_admin):
        return self.CHOICES

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(name=self.value())


class LogInline(admin.TabularInline):
    model = OperationLog
    fields = ('link_name', 'comps_status', 'started', 'finished',
              'pretty_arguments', 'type', 'initiator', 'restart_button')
    readonly_fields = ('link_name', 'comps_status', 'started', 'finished',
                       'pretty_arguments', 'type', 'restart_button', 'initiator')
    extra = 0
    can_delete = False
    fk_name = 'root'
    ordering = ('started', )


class OperationLogAdmin(admin.ModelAdmin):
    list_display = ('short_id', 'name', 'comment', 'started', 'user', 'list',
                    'initiator', 'comps_status')
    exclude = ('messages', 'name', 'status', 'arguments', 'comp_status')
    readonly_fields = ('name', 'comps_status', 'type', 'task_id', 'logs',
                       'comment', 'user', 'list', 'initiator', 'root', 'parent',
                       'retry_parent', 'started', 'finished')
    list_filter = (StatusFilter, ObsoleteFilter, InitiatorFilter,
                   OperationFilter, RootFilter)
    search_fields = ('name', '^user__email', '^list__name', 'initiator', 'comment')
    # date_hierarchy = 'started'
    inlines = [LogInline]

    change_form_template = 'admin/ml/view_operationlog.html'
    actions = ('force_retry', 'mark_as_obsolete')

    def changelist_view(self, request, extra_context=None):
        if not 'is_root' in request.GET:
            q = request.GET.copy()
            q['is_root'] = 'root'
            request.GET = q
            request.META['QUERY_STRING'] = request.GET.urlencode()
        return super(OperationLogAdmin, self).changelist_view(
            request, extra_context=extra_context)

    def force_retry(self, request, queryset):
        retry_operations(queryset, initiator=request.user.username, force=True)
        messages.info(request, u"Выбранные операции полностью перезапущены")

    force_retry.short_description = u"Перезапустить"

    def mark_as_obsolete(self, request, queryset):
        queryset.update(status='OBSOLETE')

    mark_as_obsolete.short_description = u"Пометить устаревшими"

admin.site.register(OperationLog, OperationLogAdmin)


class CorporateDomainAdmin(admin.ModelAdmin):
    list_display = ("domain", "mx_type", "is_altdomain", "created_at", "modified_at", "is_pdd", "is_type9")
    actions = ("create_in_pdd")

    def create_in_pdd(self, request, queryset):
        from mlcore.tasks.low_level import pdd_register_domain
        for d in queryset.select_related():
            try:
                pdd_register_domain.delay(context="Добавляем домен в pdd через админку", domain=d.domain)
            except Exception as err:
                messages.error(request, u"Ошибка: %s" % err)

    create_in_pdd.short_description = _("Corporate domains will be created in PDD")

admin.site.register(CorporateDomain, CorporateDomainAdmin)


class LoginRedirectAdmin(admin.ModelAdmin):
    actions = None
    list_display = ("staff", "redirect_to", )
    ordering = ("staff__login",)
    search_fields = ("staff__login", "redirect_to__login")

    def __init__(self, *args, **kwargs):
        super(LoginRedirectAdmin, self).__init__(*args, **kwargs)
        self.list_display_links = (None, )

    def has_add_permission(self, request):
        return False

admin.site.register(LoginRedirect, LoginRedirectAdmin)

class CreateListLockAdmin(admin.ModelAdmin):
    search_fields = ("name",)

admin.site.register(CreateListLock, CreateListLockAdmin)


# register_view - это из ml.admin.AdminSitePlus
admin.site.register_view('api/mlapi', name='Запрос к mlapi', view=class_views.admin_api.mlapi_admin_view)
