# -*- coding: utf-8 -*-
import logging

from django.template import RequestContext
from django.http import HttpResponse, Http404
from django.shortcuts import render_to_response, redirect, get_object_or_404
from mlcore.permissions.grant.user import is_user_owner
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.utils.translation import ugettext as _

from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList
from mlcore.ml.forms import AutoreplyProperties
from mlcore.interaction.yfurita import Autoreply, YFuritaError
from mlcore.interaction.ymailsettings import YMailSettings, YMailSettingsError
from mlcore.utils import blackbox_util
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext
from mlcore.ml.models import CorporateDomain

from django.conf import settings
dry_run = settings.DRY_RUN

logger = logging.getLogger(__name__)

FIELD1 = {
    'nospam': u'nospam',
    'from': u'От кого',
    'to': u'Кому',
    'cc': u'Копия',
    'subject': u'Тема',
}

FIELD2 = {
    3: u'содержит',
    4: u'не содержит',
}


class AutoreplySettings(BaseView):
    """ Создаем новый автоответ с условиями"""

    def __init__(self):
        super(BaseView, self).__init__()
        self.autoreply = Autoreply()
        self.request = None
        self.maillist = None
        self.passport_name = None

    @staticmethod
    def get_text_params(autoreplies):
        text_autoreplies = []
        for i, a in enumerate(autoreplies):
            text_autoreplies.append(a)
            text_autoreplies[i]['field1'] = FIELD1.get(a['field1'], '')
            text_autoreplies[i]['field2'] = FIELD2.get(a['field2'], '')
        return text_autoreplies

    def _get_bb_uid(self):
        is_yt = not self.maillist.email or '@' not in self.maillist.email
        if not is_yt:
            list_domain = self.maillist.email.split('@')[-1]
            domain = CorporateDomain.objects.get(domain=list_domain)
            is_yt = not domain.is_pdd
        return blackbox_util.get_uid(
            self.passport_name if is_yt else self.maillist.email,
            is_yateam=is_yt
        ), is_yt

    def _get_prev_from_name(self):
        uid, is_yt = self._get_bb_uid()
        mailsettings = YMailSettings(is_yateam=is_yt)
        try:
            from_name = mailsettings.get_profile_from_name(uid)
        except YMailSettingsError as err:
            logger.error("From-filed error: %s" % err)
            from_name = ''
        return from_name

    def _set_from_name(self, cleaned_data, prev_from_name):
        from_name = cleaned_data['from_name']
        if from_name == prev_from_name:
            messages.info(self.request, _('The <FROM>-field did NOT changed'))
            return
        try:
            uid, is_yt = self._get_bb_uid()
            ymailsettings = YMailSettings(is_yateam=is_yt)
            ymailsettings.update_profile(uid, from_name)
            messages.info(self.request, _('The <FROM>-field changed'))
        except YMailSettingsError:
            messages.error(self.request, _('The <FROM>-field raise exception'))

    def _set_text_autoreply(self, cleaned_data):
        field1, field3 = cleaned_data['field1'], cleaned_data['field3']
        f2 = cleaned_data['field2']
        field2 = int(f2) if f2 else ''
        text_answer = cleaned_data['text_answer']
        if text_answer:
            self.autoreply.create(self.maillist, text_answer, field1, field2, field3)
            messages.info(self.request, _('The autoreply will be add'))
        else:
            messages.error(self.request, _('The autoreply could not add. ML can not add autoreply with empty message'))

    @method_decorator(login_required)
    def __call__(self, request, ml_name):
        self.request = request
        maillist = get_object_or_404(MailList, name=ml_name)
        self.maillist = maillist

        if not is_user_owner(maillist, request.user):
            return HttpResponse("Forbidden", mimetype="text/plain", status=403)

        ytobj = YandexTeamBackendContext.objects.filter(maillist=maillist).first()
        if not ytobj:
            raise Http404
        self.passport_name = ytobj.passport_name

        try:
            a_replies = self.autoreply.detailed(maillist)
        except YFuritaError as err:
            logging.error(str(err))
            messages.error(request, _('The autoreplies is not available temporary'))
            return redirect('list', ml_name)

        prev_from_name = self._get_prev_from_name()
        if request.method == 'POST':
            f = AutoreplyProperties(request.POST)
            if f.is_valid():
                self._set_text_autoreply(f.cleaned_data)
                self._set_from_name(f.cleaned_data, prev_from_name)
                return redirect('autoreply', ml_name)
            else:
                messages.error(request, _('Wrong properties'))
        else:
            f = AutoreplyProperties(initial={
                'field1': '',
                'field2': '',
                'field3': '',
                'text_answer': '',
                'from_name': prev_from_name,
            })
        return render_to_response(
            'mailarchive/autoreply_properties.html',
            {
                'list': maillist,
                'user': request.user,
                'form': f,
                'autoreplies': self.get_text_params(a_replies),
            },
            RequestContext(request)
        )
