# -*- coding: utf-8 -*-

from django.shortcuts import render_to_response
from django.template import loader, RequestContext
from django.http import HttpResponseForbidden, HttpResponseServerError

from mlcore.ml.models import Subscribers
from mlcore.permissions.grant.user import is_user_owner


class BaseView(object):

    def _http_server_error(self, request, error_msg, template_name='500.html'):
        t = loader.get_template(template_name)
        return HttpResponseServerError(t.render(RequestContext(
            request,
            {
                'request_path': request.path, 'error_msg': error_msg,
            }
        )))

    def _http_forbidden(self, request, lst, template_name='403.html'):
        t = loader.get_template(template_name)
        return HttpResponseForbidden(t.render(RequestContext(request,
                {'request_path': request.path, 'list': lst,
                'letters': self._get_letters()})))

    def _list_marked_as_deleted(self, request, lst):
        return self._http_forbidden(request, lst,
                template_name='mailarchive/403_list_marked_as_deleted.html')

    def _403_no_permission_for_list(self, request, lst):
        return self._http_forbidden(request, lst,
                template_name='mailarchive/403_no_permission_for_list.html')

    def _403_owner_is_required(self, request, lst):
        return self._http_forbidden(request, lst,
                template_name='mailarchive/403_owner_is_required.html')

    def _get_last_viewed_list(self, request):
        return request.session.get('last_viewed_list', None)

    def _archive_not_supported(self, request, maillist):
        return render_to_response('mailarchive/archive_not_supported.html',
            RequestContext(request,
                {'request_path': request.path,
                'list': maillist}))

    def _get_letters(self):
        # Code moved to ml.templatetags.ml_tags.letters
        return []

    def _get_first_owner(self):
        if self.list.owner_set.count() == 1:
            owner = self.list.owner_set.all()[0]
        else:
            owner = None

        return owner

    def _is_user_owner(self, user=None):
        return is_user_owner(self.list, user or self.user)

    def _get_subscription(self):
        if self.user.is_authenticated():
            try:
                subscription = self.user.subscriptions.get(list=self.list)
            except Subscribers.DoesNotExist:
                subscription = None

            return subscription
        return None
