# -*- coding: utf-8 -*-
import logging

from django.template import RequestContext
from django.shortcuts import render_to_response, redirect
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.utils.translation import ugettext as _
from django.conf import settings

from mlcore.ml.class_views.base import BaseView
from mlcore.ml.forms import CreateRequestForm
from mlcore.ml.models import CorporateDomain, CreateListLock
from mlcore.tasks import operations
from mlcore.utils.domain_name import Domain, DomainError
from ya_directory import DirectoryAPIError
from ya_directory import api as directory_api

logger = logging.getLogger(__name__)


def create_domain(name):
    """
    Подключить домен к ML.

    :param name: имя домена
    :return: [<CorporateDomain>, <error_string>]
    """

    domain = Domain(name)
    try:
        domain.validate()
    except DomainError as exc:
        return None, _(exc.description)

    if domain.is_pdd_mx or domain.has_no_mx:
        # MX указывает на большую почте, вероятно к ПДД
        # Можно привязать домен к ПДД
        mx_type = CorporateDomain.MX_PDD
        directory_operations = directory_api.DirectoryMaillistsOperations()
        err_msg = None
        try:
            org_id = directory_operations.get_or_create_organization_with_domain_occupied(domain.name, check_org=True)
        except directory_api.NoAccessToOrganization as exc:
            org_id = exc.org_id
            err_msg = u'Не хватает доступа в организацию {org_id} с доменом {domain}'
        except directory_api.MaillistServiceDisabled as exc:
            org_id = exc.org_id
            err_msg = u'В организации {org_id} с доменом {domain} отключен сервис рассылок'
        except DirectoryAPIError as exc:
            logger.exception('Failed to create org with domain %s', name)
            return None, _(u'Яндекс.Коннект вернул ошибку: %(code)s') % {'code': exc.message}

        if err_msg:
            return None, _(err_msg.format(org_id=org_id, domain=domain))

    elif domain.is_corp_mx:
        # MX прибит к корповой почте
        mx_type = CorporateDomain.MX_CORP
    else:
        return None, _('Domain %s bad' % domain.name)  # на всякий случай

    doc = {'domain': name, 'mx_type': mx_type, 'is_altdomain': domain.is_altdomain, 'comment': '<created automatically>'}
    r = CorporateDomain(**doc)
    r.save()
    return r, ''


def get_domain(email):
    """
    Можно ли создать рассылку с таким доменом по данным ML.

    :param email: email рассылки
    :return: [<CorporateDomain>, <error_string>]
    """

    domain_name = email.split('@', 1)[-1]
    try:
        # Если домен существует, то ок
        domain = CorporateDomain.objects.get(domain=domain_name.lower())
        return domain, ""
    except CorporateDomain.DoesNotExist:
        # Если домен не существует - попробуем добавить его
        return create_domain(name=domain_name.lower())


class CreateRequest(BaseView):

    @method_decorator(login_required)
    def __call__(self, request):
        context = {
            'user': request.user,
            'page': 'create_request',
            'letters': self._get_letters(),
            'form': None}

        if not request.user.is_authenticated():
            return render_to_response('mailarchive/create_request.html',
                                      context, RequestContext(request))

        if request.method == 'POST':
            form = CreateRequestForm(request.POST)
            staff = request.user.staff
            if form.is_valid():
                logger.info("%s submitted create request for new list: %s",
                            staff.login, form.cleaned_data['name'])

                # проверяем домен у себя в ML
                domain, error = get_domain(form.cleaned_data['name'])

                if not domain:
                    messages.error(request, error)
                    context['form'] = form
                    return render_to_response('mailarchive/create_request.html',
                                              context, RequestContext(request))

                # TODO: мб теперь это можно делать синхронно?
                operations.create_maillist.delay(
                    {
                        'initiator': request.user.username,
                        'comment': form.cleaned_data['name']
                    },
                    data=form.cleaned_data,
                    user=staff,
                    use_cmail=False
                )

                messages.success(
                    request,
                    _(u"MSG.REQUEST_TO_CREATE_LIST %(list_name)s") %
                    {'list_name': form.cleaned_data['name']})
                CreateListLock.objects.create(name=form.cleaned_data['name'])
                return redirect('staff', staff.login)
            else:
                # form not valid
                context['form'] = form
                return render_to_response('mailarchive/create_request.html',
                                          context, RequestContext(request))

        else:
            initial = {'is_open': 1, 'is_internal': 1,
                       'archive': 0, 'before_clean': 0,
                       'imap': True,
                       'responsible': request.user.username}

            form = CreateRequestForm(initial=initial)

        context['form'] = form
        return render_to_response('mailarchive/create_request.html',
                                  context, RequestContext(request))
