# -*- coding: utf-8 -*-
import logging

from django.template import RequestContext
from django.http import HttpResponse
from django.shortcuts import render_to_response, redirect, get_object_or_404
from mlcore.permissions.grant.user import is_user_owner
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.utils.translation import ugettext as _

from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList, Owner, ARCHIVING_TYPE
from mlcore.ml.forms import ListProperties, ListOtherProperties, OwnerProperties
from mlcore.interaction.ymail import YmailBackend
from mlcore.utils.getters import get_user
from mlcore.tasks import operations

from django.conf import settings
dry_run = settings.DRY_RUN

logger = logging.getLogger(__name__)
ymail = YmailBackend()


def common_init(self, request, ml_name):
    self.request = request
    self.user = request.user
    self.list = get_object_or_404(MailList, name=ml_name)
    if self.list.is_deleted:
        return self._list_marked_as_deleted(request, self.list)

    if not self._is_user_owner():
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    return None


class EditListProperties(BaseView):

    @method_decorator(login_required)
    def __call__(self, request, ml_name):
        res = common_init(self, request, ml_name)
        if res:
            return res

        maillist = self.list
        context = {
            'list': maillist,
            'user': self.user,
            'page': 'edit_list_properties',
            'form': None,
        }

        if not request.user.is_authenticated():
            return render_to_response('mailarchive/edit_list_properties.html',
                                      context, RequestContext(request))

        if not is_user_owner(maillist, request.user):
            return HttpResponse("Forbidden", mimetype="text/plain", status=403)

        if request.method == 'POST':
            f = ListProperties(request.POST)

            if f.is_valid():
                is_internal = bool(int(f.cleaned_data['is_internal']))
                is_open = bool(int(f.cleaned_data['is_open']))

                to_close = to_open = False
                if maillist.is_open and is_open is False:
                    logger.info(u"Рассылка становится закрытой.")
                    to_close = True
                if not maillist.is_open and is_open:
                    logger.info(u"Рассылка становится открытой")
                    to_open = True

                maillist.is_internal = is_internal
                maillist.is_open = is_open

                maillist.info = f.cleaned_data['info']
                maillist.info_en = f.cleaned_data['info_en']
                maillist.info_modified = True
                maillist.external_staff_can_read = bool(int(f.cleaned_data['is_can_read_ext']))
                maillist.set_as_modified()
                maillist.save()

                if to_close or to_open:
                    ctx = {'initiator': request.user.username,
                            'comment': 'internal: %s, open: %s' % (is_internal, is_open)}
                    operations.change_maillist_open_property.delay(ctx, maillist, to_close=to_close, to_open=to_open)
                    messages.info(request, _(u'Свойства рассылки изменены, пользователи будут переподписаны через несколько минут.'))
                else:
                    messages.info(request, _(u'Свойства рассылки изменены.'))
                # TODO: delete me
                return redirect('list', ml_name)
            else:
                messages.error(request, _('Wrong properties'))
                context['form'] = f
                return render_to_response('mailarchive/edit_list_properties.html',
                                      context, RequestContext(request))
        else:
            initial = {
                'changed_name': self.list.name,
                'is_open': int(self.list.is_open),
                'is_internal': int(self.list.is_internal),
                'info': self.list.info,
                'info_en': self.list.info_en,
                'is_can_read_ext': self.list.external_staff_can_read,
            }
            f = ListProperties(initial=initial)

        context['form'] = f
        return render_to_response('mailarchive/edit_list_properties.html',
                                  context, RequestContext(request))


class EditListOtherProperties(BaseView):

    def initial(self):
        lst = self.list
        return {
            'archiving_type': lst.archiving_type if lst.is_archive else '',
            'before_clean': int(lst.before_clean) if lst.before_clean else 0,
            'max_size': int(lst.max_size) if lst.max_size else 0,
            'parent': lst.parent.fid if lst.parent else None,
        }

    @method_decorator(login_required)
    def __call__(self, request, ml_name):
        res = common_init(self, request, ml_name)
        if res:
            return res

        maillist = self.list
        context = {
            'list': self.list,
            'user': self.user,
            'page': 'edit_other_list_properties',
            'form': None,
        }

        if not request.user.is_authenticated():
            return render_to_response('mailarchive/edit_other_list_properties.html',
                                      context, RequestContext(request))

        if not is_user_owner(maillist, request.user):
            return HttpResponse("Forbidden", mimetype="text/plain", status=403)

        if request.method == 'POST':
            f = ListOtherProperties(request.POST)
            if f.is_valid():

                # Изменение родителя
                parent_maillist = f.cleaned_data['parent']  # type(parent_maillist) == Maillist
                if parent_maillist and not is_user_owner(parent_maillist, request.user):
                    messages.error(request, _('Set parent failed: You are not owner for parent maillist.'))
                    f = ListOtherProperties(initial=self.initial())
                    context['form'] = f
                    return render_to_response('mailarchive/edit_other_list_properties.html',
                                      context, RequestContext(request))

                if parent_maillist:
                    operations.change_parent.delay(
                        {
                            'initiator': request.user.username,
                            'comment': 'Изменили родителя: (parent)%s -> (child)%s' % (maillist, parent_maillist),
                        },
                        maillist,
                        self.user,
                        parent_maillist,
                    )
                    maillist.parent = parent_maillist
                    messages.info(request, _('Parent will be changed to %(parent)s.') % {'parent': parent_maillist.email})
                else:
                    messages.info(request, _('Parent will NOT be changed.'))

                # Изменение параметров архивации-удаления писем
                is_archive = True if f.cleaned_data['archiving_type'] != '' else False
                before_clean = f.cleaned_data['before_clean'] if is_archive else None
                before_clean = int(before_clean) if before_clean else None
                max_size = f.cleaned_data['max_size'] if is_archive else None
                max_size = int(max_size) if max_size else None
                # по умолчанию archiving_type == 'clean'
                archiving_type = f.cleaned_data['archiving_type'] if is_archive else ARCHIVING_TYPE.CL
                # Вызываем таску про архивирование
                # только если произошли какие-то изменения в соотв. полях базы
                if maillist.is_archive != is_archive \
                        or maillist.before_clean != before_clean \
                        or maillist.max_size != max_size \
                        or maillist.archiving_type != archiving_type:

                    operations.change_archiving_options.delay(
                        {
                            'initiator': request.user.username,
                            'comment': 'is_archive: %s, before_clean: %s, max_size: %s, archiving_type: %s' %
                                       (is_archive, before_clean, max_size, archiving_type)
                        },
                        maillist,
                        self.user,
                        is_archive,
                        before_clean,
                        max_size,
                        archiving_type,
                    )
                    maillist.is_archive = is_archive
                    maillist.before_clean = before_clean
                    maillist.max_size = max_size
                    maillist.archiving_type = archiving_type

                    messages.info(request, _('Archving-options will be change.'))
                else:
                    messages.info(request, _('Archving-options will NOT be change.'))

                maillist.save()
                return redirect('list', ml_name)
            else:
                messages.error(request, _('Wrong properties'))
                context['form'] = f
                return render_to_response('mailarchive/edit_other_list_properties.html',
                                          context, RequestContext(request))
        else:
            f = ListOtherProperties(initial=self.initial())
            context['form'] = f
            return render_to_response('mailarchive/edit_other_list_properties.html',
                                      context, RequestContext(request))


class EditOwnerProperties(BaseView):

    @method_decorator(login_required)
    def __call__(self, request, ml_name):
        res = common_init(self, request, ml_name)
        if res:
            return res

        maillist = self.list
        context = {
            'list': maillist,
            'user': request.user,
            'page': 'edit_owner_properties',
            'form': None,
        }

        if not request.user.is_authenticated():
            return render_to_response('mailarchive/edit_owner_properties.html',
                                      context, RequestContext(request))

        if not is_user_owner(maillist, request.user):
            return HttpResponse("Forbidden", mimetype="text/plain", status=403)

        if request.method == 'POST':
            f = OwnerProperties(request.POST)
            if f.is_valid():

                owner_names = f.cleaned_data['responsible']
                for owner in owner_names:
                    owner_name = owner.strip(', ')
                    try:
                        is_user = get_user(owner)
                    except Exception as ex:
                        messages.error(request, _(u"Exception: %(ex)s")%{"ex": ex})
                        context['form'] = f
                        return render_to_response('mailarchive/edit_owner_properties.html',
                                                  context, RequestContext(request))

                    if not is_user:
                        messages.error(request,
                                       _(u"You can't add the owner to this maillist"))
                        context['form'] = f
                        return render_to_response('mailarchive/edit_owner_properties.html',
                                                  context, RequestContext(request))

                    Owner.objects.get_or_create(list=maillist, user=is_user)
                    messages.success(request, _(u"Owner: %(user)s was added.") % {'user': owner_name})

                return redirect('list', ml_name)
            else:
                messages.error(request, _('Wrong properties'))
                context['form'] = f
                return render_to_response('mailarchive/edit_owner_properties.html',
                                          context, RequestContext(request))
        else:
            form = OwnerProperties()

        context['form'] = form
        return render_to_response('mailarchive/edit_owner_properties.html',
                                  context, RequestContext(request))
