# -*- coding: utf-8 -*-
import logging

from django.template import RequestContext
from django.http import HttpResponse
from django.shortcuts import render_to_response, redirect, get_object_or_404
from mlcore.permissions.grant.user import is_user_owner
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.utils.translation import ugettext as _

from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList, DirectForwards
from mlcore.ml.forms import DirectForwardForm
from mlcore.tasks.base import create_direct_forward

from django.conf import settings
dry_run = settings.DRY_RUN

logger = logging.getLogger(__name__)


class ForwardSettings(BaseView):
    """ Управление пересылкой писем """

    def __init__(self):
        super(BaseView, self).__init__()
        self.request = None
        self.maillist = None

    @method_decorator(login_required)
    def __call__(self, request, ml_name):
        self.request = request
        maillist = get_object_or_404(MailList, name=ml_name)
        self.maillist = maillist

        if not is_user_owner(maillist, request.user):
            return HttpResponse('Forbidden', mimetype='text/plain', status=403)

        if request.method == 'POST':
            f = DirectForwardForm(request.POST)
            if f.is_valid():
                email = f.cleaned_data['email_field']
                rec, created = DirectForwards.objects.get_or_create(maillist=self.maillist, email=email)
                if created or rec.is_deleted:
                    if not created:
                        rec.is_deleted = False
                        rec.status = DirectForwards.STATUS_APPROVED
                        rec.save()
                    task_context = {
                        'initiator': self.request.user.username,
                        'comment': 'Create direct forward for %s' % email,
                    }
                    create_direct_forward(task_context, rec.id)
                    messages.info(request, _(u'Forward will be created'))
                else:
                    messages.error(request, _(u'Forward already exist'))
                return redirect('forward', ml_name)
            else:
                messages.error(request, _(u'Wrong email'))
        else:
            f = DirectForwardForm()

        return render_to_response(
            'mailarchive/forward_properties.html',
            {
                'list': maillist,
                'user': request.user,
                'form': f,
                'forwards': DirectForwards.objects.filter(maillist=maillist, is_deleted=False),
            },
            RequestContext(request)
        )
