# -*- coding: utf-8 -*-
import logging

from django.shortcuts import render_to_response, get_object_or_404, redirect
from django.contrib import messages
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.template import RequestContext
from django.utils.translation import ugettext as _

from mlcore.permissions import revoke

from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList

logger = logging.getLogger(__name__)


class GroupRevoke(BaseView):

    def process_groups_info(self, group_ids, context):
        context.update(revoke.process_groups_info(self.list, group_ids))

    def perform_group_revoke(self, group_ids, context_, reason=None):
        try:
            revoke.revoke_group_permissions(self.list, group_ids,
                            by_owner=self.user.intranet_profile(), reason=reason)
        except revoke.NoPermissionsToRevokeError:
            messages.error(self.request, _("Selected permissions were not found in the database. Perhaps, they were already revoked."))
                #Выбранные доступы не найдены в базе. Возможно, они уже были удалены.")
        else:
            messages.info(self.request, _("Successfully revoked permissions for selected groups"))

    @method_decorator(login_required)
    def __call__(self, request, name):
        self.request = request
        self.user = request.user
        self.list = get_object_or_404(MailList, name=name)

        if self.list.is_deleted:
            return self._list_marked_as_deleted(request, self.list)

        self.is_owner = self._is_user_owner()
        if not self.is_owner:
            return self._403_owner_is_required(request, self.list)

        context = {
                    "list": self.list,
                    "letters": self._get_letters(),
                    "letter": self.list.letter.lower(),
                    "is_owner": self.is_owner,
                    "section": "subs",
        }

        if request.method == 'GET':
            messages.error(request, _("GET request to confirmation page is unsupported"))
            return redirect("list_subs", self.list.name)

        group_ids = [int(key[2:]) for key in request.POST\
                        if key.startswith("g_") and key[2:].isdigit()]

        if not len(group_ids):
            messages.error(request, _("You need to choose one group at least"))
            return redirect("list_subs", self.list.name)
        elif "confirm_group_revoke" in request.POST:
            self.process_groups_info(group_ids, context)
        elif "perform_group_revoke" in request.POST:
            reason = request.POST.get("reason")
            if reason != "":
                self.perform_group_revoke(group_ids, context, reason=reason)
                return redirect("list_subs", self.list.name)
            else:
                messages.error(self.request, _("You must specify a reason to revoke access"))
                self.process_groups_info(group_ids, context)
        else:
            messages.error(request, _("Internal error: unknown action"))
            return redirect("list_subs", self.list.name)

        if not context.get("group_permissions_count"):
            messages.error(request, _("Selected permissions were not found in the database. Perhaps, they were already revoked."))
                    #Выбранные доступы не найдены в базе. Возможно, они уже были удалены.")
            return redirect("list_subs", self.list.name)

        return render_to_response("mailarchive/group_revoke.html",
                RequestContext(request, context))
