# -*- coding: utf-8 -*-
import logging

from django.shortcuts import render_to_response, get_object_or_404
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.http import HttpResponse
from django.template import RequestContext
from django.utils.translation import ugettext as _

from mlcore.permissions.utils import groups_permitted_for_list, can_read, can_write
from mlcore.permissions.utils import is_external_staff
from mlcore.stats.models import GroupMembershipLog
from mlcore.ml.models import MailList
from mlcore.ml.class_views.base import BaseView
from mlcore.ymail.models import get_threads, ThreadList as TL
from mlcore.tasks import operations, base
from mlcore.ml.forms import SubscribeMailList
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext


logger = logging.getLogger(__name__)


class ShowList(BaseView):

    def get_last_joined(self, limit=5):
        last_joined_users = [(s.created_at, 'user', s.user, None)
                             for s in self.list.subscribers_set.filter(user__is_active=True).select_related('user')
                                          .order_by('-created_at')[:limit]]
        last_subscribed_emails = [(s.created_at, 'plain', s.email, None)
                                  for s in self.list.emailsubscriber_set.order_by('-created_at')[:limit]]
        last_joined = sorted(last_joined_users + last_subscribed_emails, reverse=True)[:limit]
        return last_joined

    def get_group_changes(self, offset=0, limit=10):
        return GroupMembershipLog.objects.filter(
            group__in=groups_permitted_for_list(self.list)
        ).order_by('-datetime')[offset:offset + limit]

    @method_decorator(login_required)
    def __call__(self, request, name_or_email, **kwargs):
        self.request = request
        self.user = request.user
        if '@' in name_or_email:
            self.list = get_object_or_404(MailList, email=name_or_email)
        else:
            self.list = get_object_or_404(MailList, name=name_or_email)

        if self.list.is_deleted:
            return self._list_marked_as_deleted(request, self.list)

        self.subscription = self._get_subscription()
        self.is_owner = self._is_user_owner()

        if self.user.is_authenticated():
            request.session["last_viewed_list"] = self.list.name

        if self.user.is_authenticated() and "ask_permission" in request.POST:
            operations.ask_permission.delay(
                {
                    'comment': u'Запрос доступа без подписки',
                    'initiator': request.user.username
                },
                self.user,
                self.list,
                stype=None,
                perm_type=request.POST['ask_permission']
            )

            return HttpResponse("OK", mimetype="text/plain")

        can_rd = can_read(self.user, self.list)
        if can_rd:
            last_threads = get_threads(self.list, 0, results_on_page=5)
        else:
            last_threads = TL([], 0, False)

        context = {
            "list": self.list,
            "letters": self._get_letters(),
            "letter": self.list.letter.lower(),
            "subscription": self.subscription,
            "owner": self._get_first_owner(),
            "is_owner": self.is_owner,
            "is_staff": self.user.is_staff,
            "max_size": self.list.max_length / 1024,
            "section": "desc",
            "last_joined": self.get_last_joined(),
            "group_changes": self.get_group_changes(),
            "last_threads": last_threads,
            "can_read": can_rd,
            "can_write": can_write(self.user, self.list),
            "has_external_subsribers": self._has_external_subscribers(),
            "is_tracking_list": self._is_tracking_list(),
            "subscribe_maillists_form": SubscribeMailList(),
        }

        return render_to_response('mailarchive/list.html', context,
                                  RequestContext(request))

    def _has_external_subscribers(self):

        has_external_subsribers = False

        # определяем, есть ли в рассылке внешние сотрудники через признак Staff.affiliation
        ## Это всё тормозит как не в себя. TODO: переделать определение наличия внешний сотрудников
        # external_subscribers = False # [s for s in self.list.subscribers_set.all() if is_external_staff(s.user)]
        # has_external_subsribers |= bool(external_subscribers)

        # смотрим email-подписчиков с внешними emailами
        has_external_subsribers |= self.list.emailsubscriber_set.exclude(
            email__iregex='^[a-z_-]+-(jira|bugs|task)@'
        ).exclude(
            email__icontains='yandex-team'
        ).exclude(
            email__iregex=r'[a-z0-9\-\.]+@(yaprobki|yamoney)\.[a-z\.]+'
        ).exists()

        return has_external_subsribers

    def _is_tracking_list(self):
        tracking_emails = self.list.emailsubscriber_set.filter(
            email__iregex=r'^[a-z_-]+-(jira|bugs|task)@'
        )
        return tracking_emails.exists()

    def _has_subscribers(self):
        return bool(list(self.list.emailsubscriber_set.all()) + list(self.list.subscribers_set.all()))