# -*- coding: utf-8 -*-
import json


from django.shortcuts import render_to_response, get_object_or_404
from django.contrib import messages
from django.db.models import Count
from django.contrib.auth.models import User
from django.contrib.auth.decorators import login_required
from django.http import HttpResponseRedirect, HttpResponse
from django.template import RequestContext, loader, Context
from django.utils.decorators import method_decorator
from django.utils.translation import ugettext as _

from django_intranet_stuff.models import Group

from mlcore.permissions.signals import (permissions_granted, group_permission_granted)
from mlcore.permissions.models import ListPermission, Type, GroupPermission
from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList, EmailSubscriber, Subscribers
from mlcore.tasks.operations import unsubscribe_and_notify, unsubscribe_email
from mlcore.permissions.utils import can_read
from mlcore.ml.forms import UnsubscribeForm
from mlcore.utils.getters import get_user
from django.core.exceptions import ObjectDoesNotExist


SERVICE_EMAILS = ['otrs@otrs-imap.yandex-team.ru', 'otrs@otrs-inbox.yandex-team.ru', 'separator@separator.yandex-team.ru',
                  'tcrm@yandex-team.ru', 'tcrmlite@yandex-team.ru']


class ListSubscribers(BaseView):

    def __init__(self):
        self.list = None  # рассылка
        self.user = None  # пользователь на странице
        self.is_owner = None  # ответсвенный ли за рассылку пользователь на странице
        self.request = None  # request
        self.inbox = None  # количество подписчиков во входящие
        self.imap = None  # количество подписчиков в общую папку
        self.both = None  # количество подписчиков у данной рассылки
        self.separate = None  # количество подписчиков в отдельную папку
        self.subscription = None  # список всех подписчиков
        super(ListSubscribers, self).__init__()

    def regroup_groups_by_type(self, groups):
        splitted = {'departments': [], 'services': [], 'intra': []}
        for group in groups:
            stypes = GroupPermission.objects.filter(
                group__pk=group.id,
                list__pk=self.list.id
            ).values_list('type__name', flat=True)

            if group.service_id:
                splitted['services'].append((group, stypes))
            elif group.department_id:
                splitted['departments'].append((group, stypes))
            else:
                splitted['intra'].append((group, stypes))
        return splitted

    def get_subscribers_list(self):
        user_subscriptions_base = self.list.subscribers_set.filter(user__is_active=1)
        email_subscriptions = self.list.emailsubscriber_set.all()
        counters = dict((x['stype'], x['count'])
                        for x in user_subscriptions_base.values("stype").annotate(count=Count("id")))

        self.inbox = email_subscriptions.count() + counters.get("inbox", 0)
        self.imap = counters.get("imap", 0)
        self.both = counters.get("both", 0)
        self.separate = counters.get("separate", 0)

        user_subscriptions = user_subscriptions_base.select_related("user__staff").order_by(
            "user__last_name",
            "user__first_name",
        ).all()

        return {
            'user': user_subscriptions,
            'email': email_subscriptions,
            'united': list(user_subscriptions) + list(email_subscriptions)
        }

    def get_permitted_groups(self):
        return Group.objects.filter(permissions__list=self.list).distinct()

    def get_readers(self):

        not_subscribed_owners = User.objects.filter(
            owner__list=self.list
        ).select_related('staff').filter(
            is_active=1
        ).exclude(
            subscriptions__list=self.list
        ).extra(
            select={'is_owner': 1}
        ).only('username', 'first_name', 'last_name')

        not_subscribed_but_permitted = User.objects.filter(
            is_active=1,
            permissions__list=self.list,
            permissions__approved=True
        ).select_related('staff').exclude(
            subscriptions__list=self.list
        ).only('username', 'first_name', 'last_name')

        not_subscribed_owners_set = set(not_subscribed_owners)

        for u in set(not_subscribed_but_permitted).intersection(not_subscribed_owners_set):
            u.is_permitted_owner = True

        all_in_one = list(not_subscribed_owners_set.union(not_subscribed_but_permitted))
        all_in_one.sort(key=lambda x: x.display_name())
        return all_in_one

    def revoke_permissions(self, user_ids, reason=''):
        for user in User.objects.filter(id__in=user_ids):
            unsubscribe_and_notify.delay(
                {
                    'initiator': self.user.username,
                    'comment': u"Отписка владельцем"
                },
                user,
                self.list,
                reason,
                by_owner=self.user
            )

    def is_service_emails(self, key):
        m = EmailSubscriber.objects.filter(pk=key)
        if m and m[0].email in SERVICE_EMAILS:
            return True
        return False

    def unsubscribe_emails(self, emailsub_ids):
        for emailSub in EmailSubscriber.objects.filter(pk__in=emailsub_ids):
            unsubscribe_email.delay(
                {
                    'initiator': self.user.username,
                    'comment': emailSub.email
                },
                emailSub.email,
                emailSub.list
            )

    def process_access_revoke(self):
        if self.is_owner:
            reason = self.request.REQUEST.get('reason', '')
            is_service_emailsub = False
            if reason:
                user_ids = []
                email_subs_ids = []
                for key in self.request.POST:
                    if key.startswith("u_"):
                        user_ids.append(int(key[2:]))
                    if key.startswith("e_") and self.is_service_emails(int(key[2:])):
                        is_service_emailsub = True
                    elif key.startswith("e_"):
                        email_subs_ids.append(int(key[2:]))

                if is_service_emailsub:
                    messages.error(self.request, _(u"You can't unsubscribe service email"))
                elif not user_ids and not email_subs_ids:
                    messages.error(self.request, _(u"MSG.CHOOSE_ONE_SUBSCRIBER"))
                else:
                    self.revoke_permissions(user_ids, reason=reason)
                    self.unsubscribe_emails(email_subs_ids)
                    messages.success(self.request, _(u"MSG.ACCESS_REVOKED"))
            else:
                messages.error(self.request, _("You must specify a reason to revoke access"))
        else:
            messages.error(self.request, _(u"MSG.YOU_CANNOT_EDIT_RIGHTS"))
        return HttpResponseRedirect(self.request.path)

    def process_access_revoke_mini(self):
        return_page = HttpResponseRedirect(self.request.path)

        if not self.is_owner:
            messages.error(self.request, _(u"MSG.YOU_CANNOT_EDIT_RIGHTS"))
            return return_page

        reason = self.request.REQUEST.get('reason', '')
        if not reason:
            messages.error(self.request, _("You must specify a reason to revoke access"))
            return return_page

        f = UnsubscribeForm(self.request.POST)

        if f.is_valid():
            user_ids = []
            email_subs_ids = []
            user_names = f.cleaned_data['login']

            # в автокомплите содержатся только работнички
            for u in user_names:
                user_name = u.strip(', ')
                try:
                    user = get_user(user_name)
                    sub = Subscribers.objects.get(user=user, list=self.list)
                    user_ids.append(sub.user.id)
                    messages.info(self.request, _(u"User (%(user)s) will be revoked") % {"user": user_name})
                except (TypeError, ObjectDoesNotExist) as err:
                    try:
                        esub = EmailSubscriber.objects.get(email=user_name, list=self.list)
                        email_subs_ids.append(esub.id)
                        messages.info(self.request, _(u"Email (%(email)s) will be revoked") % {"email": user_name})
                    except ObjectDoesNotExist as err:
                        messages.error(self.request, _(u"No such user (%(user)s) in %(maillist)s") % {"maillist": self.list, "user": user_name})

            self.revoke_permissions(user_ids, reason=reason)
            self.unsubscribe_emails(email_subs_ids)
            messages.success(self.request, _(u"MSG.ACCESS_REVOKED"))
            return return_page

        messages.error(self.request, _('Wrong properties'))
        return return_page

    def process_access_grant(self):
        try:
            data = json.loads(self.request.POST.keys()[0])
        except (ValueError, KeyError):
            return HttpResponse(
                json.dumps({'status': 'error', 'message': _('Wrong arguments')}),
                content_type='application/json')

        users = {}
        groups = {}

        for key, prop in data.iteritems():
            if prop['type'] == 'user':
                users[key] = prop['permission']
            elif prop['type'] == 'group':
                groups[key] = prop['permission']

        types = dict(
            read=Type.objects.get(name='read'),
            write=Type.objects.get(name='write'))

        granted_users = User.objects.filter(username__in=users.iterkeys())
        for u in granted_users:
            permission = types[users[u.username]]
            ListPermission.objects.grant_permission(maillist=self.list, user=u,
                                                    permission_type=permission)

        if len(users):
            permissions_granted.send(sender=self, by_owner=self.user.staff,
                                     list=self.list, granted_users=granted_users)

        granted_groups = Group.objects.filter(url__in=groups.iterkeys())
        for g in granted_groups:
            permission = types[groups[g.url]]
            __, is_created = g.permissions.get_or_create(list=self.list,
                                                         type_id=permission.id)
            if is_created:
                group_permission_granted.send(
                    sender=None, list=self.list,
                    by_owner=self.user.staff, group_id=g.id)

        if len(granted_users) or len(granted_groups):
            messages.success(self.request, _(u"MSG.ACCESS_GRANTED"))

        return HttpResponse(json.dumps({'status': 'ok'}),
                            content_type='application/json')


class ListSubscribersHTML(ListSubscribers):

    @method_decorator(login_required)
    def __call__(self, request, name, **kwargs):

        show_subscribers_limit = 500
        self.request = request
        self.user = request.user
        self.list = get_object_or_404(MailList, name=name)
        if self.list.is_deleted:
            return self._list_marked_as_deleted(request, self.list)

        if not can_read(self.user, self.list) and not self.user.is_staff:
            return self._403_no_permission_for_list(request, self.list)

        self.subscription = self._get_subscription()
        self.is_owner = self._is_user_owner()
        if self.is_owner:
            if self.request.method == "POST" and "action_revoke" in self.request.POST:
                return self.process_access_revoke()
            if self.request.method == "POST" and "action_revoke_mini" in self.request.POST:
                return self.process_access_revoke_mini()
            if self.request.method == 'POST' and self.request.is_ajax():
                return self.process_access_grant()

        subscribers_count = self.list.subscribers_count()
        context = {
            "list": self.list,
            "letters": self._get_letters(),
            "letter": self.list.letter.lower(),
            "subscription": self.subscription,
            "owner": self._get_first_owner(),
            "is_owner": self.is_owner,
            "show_add_access_control": (self.list.readonly or
                                        (not self.list.is_open and self.is_owner)),
            "max_size": self.list.max_length / 1024,
            "section": "subs",
            "permitted_groups": self.regroup_groups_by_type(self.get_permitted_groups()),
        }
        if subscribers_count > show_subscribers_limit:
            context.update({
                "subscribers": [],
                "unsubscribe": UnsubscribeForm(),
                "subscribers_count": subscribers_count,
            })
            template_name = 'mailarchive/list_subscribers_mini.html'
        else:
            context.update({
                "subscribers": self.get_subscribers_list(),
                "inbox_count": self.inbox,
                "imap_count": self.imap,
                "both_count": self.both,
                "separate_count": self.separate,
            })
            template_name = 'mailarchive/list_subscribers.html'
        request_context = RequestContext(request, context)
        return render_to_response(template_name, request_context)


class ListSubscribersCVS(ListSubscribers):

    @method_decorator(login_required)
    def __call__(self, request, name, **kwargs):

        self.request = request
        self.user = request.user
        self.list = get_object_or_404(MailList, name=name)
        if self.list.is_deleted:
            return self._list_marked_as_deleted(request, self.list)

        if not can_read(self.user, self.list) and not self.user.is_staff:
            return self._403_no_permission_for_list(request, self.list)

        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = 'attachment; filename="subscribers_%s.csv"' % self.list.name

        subscr = self.get_subscribers_list()

        csv_data = [
            ('# Count of inbox: %d' % self.inbox, 'Imap: %d' % self.imap, 'Both: %d' % self.both,
             'Separate folder: %d' % self.separate),
            ('Username', 'Email', 'Type of subscription', 'Affiliation'),
        ]

        for s in subscr["user"]:
            csv_data.append((s.user.username, s.user.email, s.stype, s.user.staff.affiliation))

        for s in subscr["email"]:
            csv_data.append((s.email, s.email, s.getType()))

        t = loader.get_template('mailarchive/subscribers.txt')
        c = Context({
            'data': csv_data,
        })
        response.write(t.render(c))
        return response
