# -*- coding: utf-8 -*-

from django import forms
from django.conf import settings
from django.core import validators
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _
from django_intranet_stuff.forms.fields import AutocompleteField
from django_intranet_stuff.forms.widgets import AutocompleteWidget
from django_intranet_stuff.models import Staff

from mlcore.ml.models import MailList
from mlcore.utils.email_validator import MaillistEmailValidator


class MailListNameField(forms.CharField):

    def to_python(self, value):
        data = super(MailListNameField, self).to_python(value).strip().lower()
        if '@' not in data:
            data += '@yandex-team.ru'
        return data


class StaffLoginAutocompleteWidget(forms.TextInput):
    def __init__(self, *args, **kwargs):
        autocomplete_value = kwargs.pop('autocomplete_value', 'login')
        attrs = kwargs.get('attrs', {})
        attrs['class'] = attrs.get('class', '') + ' js-autocomplete_staff'

        attrs['onclick'] = (
                "return {'multiple': true, 'result': 'login', 'value': '%s'}"
                % autocomplete_value)

        kwargs['attrs'] = attrs
        super(StaffLoginAutocompleteWidget, self).__init__(*args, **kwargs)


class StaffLoginAutocompleteLargeWidget(forms.Textarea):
    def __init__(self, *args, **kwargs):
        autocomplete_value = kwargs.pop('autocomplete_value', 'login')
        attrs = kwargs.get('attrs', {})
        attrs['class'] = attrs.get('class', '') + ' js-autocomplete_staff'

        attrs['onclick'] = (
                r"return {'multiple': true, 'result': 'login', 'value': '%s', 'multipleSeparator': '\n'}"
                % autocomplete_value)

        kwargs['attrs'] = attrs
        super(StaffLoginAutocompleteLargeWidget, self).__init__(*args, **kwargs)


class StaffListField(forms.CharField):
    def __init__(self, *args, **kwargs):
        widget = kwargs.get('widget', None)
        self.separator = kwargs.pop('separator', ',')
        if widget is None:
            autocomplete_value = kwargs.pop('autocomplete_value', 'login_ld')
            widget = StaffLoginAutocompleteWidget(attrs={'class': 'b-newml__widefield'},
                                                  autocomplete_value=autocomplete_value)
        kwargs['widget'] = widget
        super(StaffListField, self).__init__(*args, **kwargs)

    def clean(self, value):
        super(forms.CharField, self).clean(value)
        splitted = filter(len, (x.strip() for x in value.split(self.separator)))
        if self.required and len(splitted) == 0:
            raise forms.ValidationError(self.error_messages['required'])
        else:
            return splitted


class RadioButtonBooleanField(forms.TypedChoiceField):
    def __init__(self, true_label='True', false_label='False', *args, **kwargs):
        kwargs['widget'] = forms.RadioSelect
        kwargs['choices'] = ((1, true_label), (0, false_label))
        kwargs['coerce'] = lambda x: bool(int(x))
        super(RadioButtonBooleanField, self).__init__(*args, **kwargs)


class CreateRequestForm(forms.Form):
    name = MailListNameField(label=_(u"LBL.LIST_NAME"),
                             widget=forms.TextInput(
                                 attrs={'style': 'width:90%', 'class': 'b-newml__big'}),
                             validators=[validators.validate_email, MaillistEmailValidator()])

    responsible = StaffListField(label=_(u"LBL.RESPONSIBLES"))
    info = forms.CharField(label=_(u"LBL.DESCRIPTION"),
                           widget=forms.Textarea(
                               attrs={'rows': 3, 'cols': 10, 'class': 'b-newml__widefield'}))
    info_en = forms.CharField(label=_(u"LBL.DESCRIPTION"),
                              widget=forms.Textarea(
                                  attrs={'rows': 3, 'cols': 10,
                                         'class': 'b-newml__widefield'}))
    is_open = RadioButtonBooleanField(_(u"LBL.OPEN"), _(u"LBL.CLOSE"))
    is_internal = RadioButtonBooleanField(_(u"LBL.INNER"), _(u"LBL.OUTER"))

    archive = forms.BooleanField(required=False, initial=False, label=_(u"Use archiving"))
    before_clean = forms.IntegerField(
        required=False, label=_(u"Days before cleaning"),
        widget=forms.TextInput(attrs={'size': '1', 'maxlength': '4'}),
        initial=0)

    @staticmethod
    def _logins_to_emails(logins):
        users = set()
        emails = set()

        for login in logins:
            if '@' in login and login.index('@') < (len(login) - 1):
                # assume that login is email itself
                emails.add(login)
            else:
                users.add(login)

        users_emails = set(Staff.objects.filter(login__in=users)
                           .values_list('work_email', flat=True))

        return emails | users_emails

    def clean_responsible(self):
        data = self.cleaned_data['responsible']
        return self._logins_to_emails(data)


class ApiCreateRequestForm(CreateRequestForm):
    dry_run = forms.BooleanField(required=False)


class MailListAutocompleteWidget(AutocompleteWidget):

    def __init__(self, attrs=None, to_field_name='name'):
        if attrs is None:
            attrs = {}

        attrs.update({
            'onclick': mark_safe("return {hidden: '%s', is_imap: true}" % to_field_name),
            'class': 'b-subscribe-friend-input js-autocomplete_maillists',
        })

        super(MailListAutocompleteWidget, self).__init__(
            attrs=attrs, to_field_name=to_field_name)

    class Media:
        js = (
            '/js/_ml.js',
            '/js/_maillists.js',
        )


class MailListField(AutocompleteField):
    class Media:
        # Приходится переопределять, потому что в DIS захардкожен http
        js = (
            '//%s/media/js/_js-autocomplete_staff.js' % settings.CENTER_MASTER_WWW,
        )

    class Meta:
        model = MailList

    def __init__(self, **kw):
        kw['to_field_name'] = 'name'
        self.widget = MailListAutocompleteWidget()
        super(MailListField, self).__init__(**kw)


class SubscribeMailList(forms.Form):
    maillist = MailListField(required=True)


class SearchForm(forms.Form):
    maillist = forms.CharField(required=True)


class FriendSubscribeForm(forms.Form):
    friends__login_ld = StaffListField(required=True, separator='\n', widget=StaffLoginAutocompleteLargeWidget)
    access_reason = forms.CharField(
        required=False,
        widget=forms.Textarea(attrs={
            'rows': 3,
            'cols': 80,
            'style': 'display',
            'requered': True,
            'maxlength': 190,
        })
    )
    has_reason = forms.BooleanField(required=False, initial=False)
    stype = forms.ChoiceField(required=True, choices=(
        ('inbox', _('In inbox')),
        ('imap', _('In imap folder')),
        ('both', _('In inbox and imap folder')),
        ('separate', _('In a separate folder'))
    ))


class UnsubscribeForm(forms.Form):
    login = StaffListField(label=u"Отписать подписчиков", required=False)


class ListProperties(forms.Form):
    """
        Форма, которая изменяет описание рассылки на русском, на английском, параметры рассылки.
    """
    is_open = RadioButtonBooleanField(_(u"LBL.OPEN"), _(u"LBL.CLOSE"))
    is_internal = RadioButtonBooleanField(_(u"LBL.INNER"), _(u"LBL.OUTER"))
    info = forms.CharField(
        label=_(u"LBL.DESCRIPTION"),
        widget=forms.Textarea(attrs={
            'rows': 3,
            'cols': 60,
            'style': 'display',
            'class': 'b-newml__widefield'
        })
    )
    info_en = forms.CharField(
        label=_(u"LBL.DESCRIPTION"),
        widget=forms.Textarea(attrs={
            'rows': 3,
            'cols': 60,
            'style': 'display',
            'class': 'b-newml__widefield'
        })
    )
    is_can_read_ext = forms.BooleanField(
        label=_(u'Разрешить подписку внешним сотрудникам'),
        required=False,
        initial=False
    )


class ListOtherProperties(forms.Form):
    """
        Форма с редко изменеяемыми свойствами: архивирование, общие флажки, родительская папка
    """
    ARCHIVING_TYPE_CHOICES = (
        ('', _(u'do not clean')),
        ('archive', _(u'archive')),
        ('clean', _(u'clean')),
    )

    parent = MailListField(label=u'Переместить в другую папку', required=False)
    archiving_type = forms.ChoiceField(label=u'Очищать рассылку?', required=False, choices=ARCHIVING_TYPE_CHOICES)
    max_size = forms.IntegerField(
        required=False, label=_(u"Max count to save"),
        widget=forms.TextInput(attrs={'size': '4', 'maxlength': '7'}),
        initial=0
    )
    before_clean = forms.IntegerField(
        required=False, label=_(u"Days before cleaning"),
        widget=forms.TextInput(attrs={'size': '3', 'maxlength': '4'}),
        initial=0
    )


class OwnerProperties(forms.Form):
    """
        Форма для изменения, удаления (смены) ответственных за рассылку
    """
    responsible = StaffListField(label=_(u"Добавить ответственного"), required=False)


class BulkUserSubscribeForm(forms.Form):
    name = forms.CharField(label=u"Имя рассылки")
    type = forms.ChoiceField(choices=(('imap', _('shared folder')),
                                      ('inbox', _('inbox'))),
                             label=u"Тип подписки")
    logins_and_mails = forms.CharField(
        widget=forms.Textarea(
            attrs={'placeholder': u'На каждой cтрочке по одному емейлу'}),
        label=u"Логины или емейлы",
        help_text=(u"Логины для сотрудников или полные емейлы для внешних "
                   u"на каждой строчке"))

    def clean_logins_and_mails(self):
        logins = self.cleaned_data['logins_and_mails']
        return logins.splitlines()


class AutoreplyProperties(forms.Form):
    """
    Форма для автоответчика с разными условиями. (ML-1009)
    Можно создать условия проверки Темы, Кому, Копия, От кого.
    Создать шаблон и проверить, содержится он или не содержится для field1.
    """
    FIELD1 = [
        ('from', _(u'От кого')),
        ('to', _(u'Кому')),
        ('cc', _(u'Копия')),
        ('subject', _(u'Тема')),
    ]
    FIELD2 = [
        ('3', u'содержит'),
        ('4', u'не содержит')
    ]
    field1 = forms.CharField(required=False, widget=forms.Select(choices=FIELD1))
    field2 = forms.CharField(required=False, widget=forms.Select(choices=FIELD2))
    field3 = forms.CharField(required=False, label=u"Шаблон для условия")
    text_answer = forms.CharField(
        widget=forms.Textarea(
            attrs={
                'placeholder': u'Текст автоответчика',
                'rows': 3,
                'cols': 60,
                'style': 'display',
                'class': 'b-newml__widefield'
            }
        ),
        label=u"Текст автоответчика",
        required=False,
    )
    from_name = forms.CharField(required=False, label=u"Поле From в письме автоответа")


class DomainListField(forms.CharField):
    def __init__(self, *args, **kwargs):
        super(DomainListField, self).__init__(*args, **kwargs)

    def clean(self, value):
        super(forms.CharField, self).clean(value)
        splitted = filter(len, (x.strip() for x in value.split(',')))
        if self.required and len(splitted) == 0:
            raise forms.ValidationError(self.error_messages['required'])
        else:
            return splitted


class DomainsForm(forms.Form):
    domains = DomainListField()


class DirectForwardForm(forms.Form):
    email_field = forms.EmailField()

    def clean_email_field(self):
        return self.cleaned_data['email_field'].lower()
