#!/usr/bin/python
# encoding: utf-8

"""
Синхронизация подписок.

Скрипт сравнивает состояние подписки в базе ml с базами cmail и ymail.
Если находит расхождения - подписывает/отписывает в cmail и ymail.

TODO:
  - убрать print, заменить на logging.debug
  - выводить логи на stderr
"""

import glob
import os.path
import json
import sys
import os

# --- /usr/lib/yandex/tools-maillists/src/maillists/manage.py

if __name__ == '__main__':

    sys.path = ['/etc/yandex/tools-maillists',
                '/usr/lib/yandex/tools-maillists/src',
                '/usr/lib/yandex/tools-maillists/src/maillists',
                '/etc/yandex/tools-maillists/celery/',
                ] + sys.path

    os.environ.setdefault("DJANGO_SETTINGS_MODULE", "settings")
    import settings
    settings.LOGGING = {'version':1}

import logging
from datetime import datetime
import time

from mlcore.interaction.ymail import YmailBackend, YmailError
from mlcore.interaction.cmail import CmailBackend, CmailError, timestamp
from mlcore.interaction.utils import dict2xml, dict2xmlstring
from lxml import etree
from mlcore.ml.models import MailList
from mlcore.ml.models import Autosubscription, Subscribers, EmailSubscriber, SuidLookup, Autosubscription
from django.contrib.auth.models import User
from mlcore.utils.getters import get_user, get_list, get_suid, get_staff, get_list_suid
from mlcore.subscribe import subscription_type
from mlcore.tasks.low_level import subscribe_cmail, subscribe_ymail, unsubscribe_cmail, unsubscribe_ymail
import django_intranet_stuff
from django.core.management.base import BaseCommand
from django.utils.encoding import smart_str
from functools import wraps
from optparse import make_option

log = logging.getLogger(__name__)

IGNORE_USERS = ['efim', ]

DEBUG = 0

class NoEmail(Exception):
    pass

def normalize_email(email):
    return email.strip().lower()


def blackbox_get_login_by_suid(suid, sid=2):
    import requests, json
    url = 'https://blackbox.yandex-team.ru:443/blackbox'
    params = { 'suid': suid, 'userip': '127.0.0.1', 'method':'userinfo', 'sid': sid, 'dbfields': 'hosts.db_id.2', 'format': 'json' }
    r = requests.get(url, params=params)
    return json.loads(r.content)['users'][0]['login']


def get_login_by_suid(suid):
    """Возвращает логин по суиду."""
    try:
        return SuidLookup.objects.get(suid=int(suid)).login
    except SuidLookup.DoesNotExist:
        login = blackbox_get_login_by_suid(suid)
        if not login:
            return

        SuidLookup.objects.create(suid=int(suid), login=login)
        return login


def memoize(func):
    @wraps(func)
    def wrap(self, *args):
        key = list(args)
        key.insert(0, func.__name__)
        key = str(key)
        cache =  getattr(self, '__memoize_cache', None)
        if cache is None:
            cache = {}
            self.__memoize_cache = cache
        if key in cache:
            r = cache[key]
        else:
            r = func(self, *args)
            cache[key] = r
        return r
    return wrap


def filter_cmail_system_addresses(lst):
    # Отфильтруем служебные адреса-подписчики в cmail
    stopwords = ('@ll.yandex-team.ru', 'procmail')

    def has_stop_word(email):
        for w in stopwords:
            if w in email:
                return True
        return False

    for email in filter(None, lst):
        if has_stop_word(email): continue
        if 'yandex' in email:
            try:
                # На всякий случай, чтобы не отписать системные email-ы, делаем еще одну проверку
                get_user(email)
            except:
                continue
        yield email


class MyYmailBackend(YmailBackend):

    """ Если в YmailBackend уже есть этот метод, то можно отпилить MyYmailBackend """

    def subscribers(self, maillist):
        """ Отдает суиды пользователей, подписанных на общую папку """
        api_name = 'Subscribers'
        folder_name = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_name})
        if self.dry_run:
            return
        return self.check_and_parse(resp)


class MyCmailBackend(CmailBackend):

    """ В классе CmailBackend был баг, поэтому делаем воркэраунд.
        Если баг исправилили, то можно отпилить MyCmailBackend
    """

    def get_subscribers(self, email):

        """ Возвращает подписчиков на рассылку по ее email'у

       { 'smtp': ['yurkennis@yandex-team.ru'] }

        """
        data = {'version': '1.0', 'timestamp': timestamp(),
                'list': email}
        body = dict2xml(data, 'request', type="get", object="maillist")
        #print __file__, "cmail url=", self.url
        resp = self.http.post(self.url, etree.tostring(body, encoding='unicode'), timeout=60)

        if self.dry_run:
            return {}

        #print resp.content
        info = self.check_and_parse(resp)

        res = {}
        for child in info.iterchildren():
            if child.tag == 'subscribers':
                sub_data = []
                for item in child.iterchildren():
                    sub_data.append(item.text)
                res[child.attrib['type']] = sub_data

        return res



class SyncerBase(object):

    """
    Базовый класс для синхронизатора.
    Умеет выдавать списки подписчиков из разных баз.
    """

    def __init__(self, maillist):
        self.maillist = maillist
        self.email = self.maillist.email
        #if not self.email:
        #    print "email is none for list", self.maillist, "skip"
        #    raise NoEmail
        self._cmail_subscribers = None
        self._ymail_subscribers = None

    @memoize
    def cmail_subscribers(self):
        try:
            p = MyCmailBackend().get_subscribers(self.email)
        except Exception, e:
            print __name__, "Error loading cmail data for list", self.email, e
            raise
        #print "cmail_subscribers=", p['smtp']
        subscribers = p['smtp']
        return subscribers and filter(None, subscribers) or []

    @memoize
    def ymail_subscribers(self):
        return [ suid for suid in MyYmailBackend().subscribers(self.maillist) ]



class SyncSubs(SyncerBase):

    help = u"Синхронизация подписок одного списка рассылки с ymail и cmail"

    def __init__(self, maillist, dry_run=True):
        super(SyncSubs, self).__init__(maillist)
        self.dry_run = dry_run
        self.ignore_cmail_errors = False

    def _run_task(self, func, **kwargs):
        task = func.si(context={}, **kwargs)
        if not self.dry_run:
            time.sleep(1)
            task = task.delay()
        else:
            pass
        print task
        return task

    def need_imap_subscribe(self, user, maillist, autosubscription=False):
        print "Need imap subscribe %s on %s" % (user.email, maillist)
        return self._run_task(subscribe_ymail, user=user, maillist=maillist, type=subscription_type.IMAP)

    def need_inbox_subscribe(self, user, maillist):
        print "Need inbox subscribe %s on %s" % (user, maillist)
        return self._run_task(subscribe_cmail, user=user, maillist=maillist, type=subscription_type.INBOX)

    def need_email_subscribe(self, email, maillist):
        print "Need email subscribe %s on %s" % (email, maillist)
        return self._run_task(subscribe_cmail, user=email, maillist=maillist, type=subscription_type.INBOX)

    def need_inbox_unsubscribe(self, email, maillist):
        print "Need inbox unsubscribe %s on %s" % (email, maillist)
        return self._run_task(unsubscribe_cmail, user=email, maillist=maillist, type=subscription_type.INBOX)

    def need_imap_unsubscribe(self, user, maillist):
        print "Need imap unsubscribe %s on %s" % (user, maillist)
        return self._run_task(unsubscribe_ymail, user=user, maillist=maillist, type=subscription_type.IMAP)


    def is_subscribed_in_cmail(self, email):
        return  (email in self.cmail_subscribers() )

    def is_subscribed_in_ymail(self, user):
        suid = get_suid(user)
        return (suid in self.ymail_subscribers() )


    def sync_imap_subscriptions(self):

        if DEBUG:
            print "sync_imap_subscriptions for", self.maillist
        users_should_be_subscribed = []

        for subs in Subscribers.objects.filter(list=self.maillist):
            if DEBUG: print "sync_imap_subscriptions", subs
            stype = subs.getType()
            if stype not in (subscription_type.BOTH, subscription_type.IMAP):
                continue
            #print "sync_imap_subscriptions", subs
            users_should_be_subscribed.append(subs.user)

            if not self.is_subscribed_in_ymail(subs.user):
                self.need_imap_subscribe(user=subs.user, maillist=self.maillist)

        # TODO: unsubscribe users in ymail
        to_unsubscribe =  set(self.ymail_subscribers()) - set([get_suid(user) for user in users_should_be_subscribed])
        if to_unsubscribe:
            print "Should unsubscribe in ymail:", to_unsubscribe
        for suid in to_unsubscribe:
            self.need_imap_unsubscribe(user=get_login_by_suid(suid), maillist=self.maillist)

    def sync_inbox_subscriptions(self):

        users_should_be_subscribed = []

        # Кто подписан в инбокс ?
        for subs in Subscribers.objects.filter(list=self.maillist):
            stype = subs.getType()
            if stype not in (subscription_type.BOTH, subscription_type.INBOX):
                continue
            users_should_be_subscribed.append(subs.user.email.strip())

        # Кто подписан на email ?
        for es in EmailSubscriber.objects.filter(list=self.maillist):
            users_should_be_subscribed.append(es.email.strip())

        # Подпишем всех, кто не подписан
        for email in filter(None, users_should_be_subscribed):
            #print "sync_inbox_subscriptions", email
            if not self.is_subscribed_in_cmail(email):
                self.need_inbox_subscribe(user=email, maillist=self.maillist)

        # Отпишем всех, кто не должен быть подписан
        users_should_be_subscribed = set(map(normalize_email, filter(None, users_should_be_subscribed)))
        real_cmail_subscribers = set(filter_cmail_system_addresses(map(normalize_email,self.cmail_subscribers())))
        to_unsubscribe = real_cmail_subscribers - users_should_be_subscribed
        if to_unsubscribe:
            print "Should unsubscribe in cmail:", to_unsubscribe
        for email in to_unsubscribe:
            self.need_inbox_unsubscribe(email=email, maillist=self.maillist)


    def sync(self):
        self.sync_imap_subscriptions()
        self.sync_inbox_subscriptions()


def iter_all_maillists(start_from=None):
    start_from_id = start_from and get_list(start_from).id or 0
    for ml in MailList.objects.filter(id__gte=start_from_id).order_by('id'):
        # нужно пропускать рассылки, у которых совпадает suid
        duplicates = MailList.objects.filter(fsuid=ml.fsuid)
        if len(duplicates)>1:
            logging.debug('maillist %s has duplicates by suid, skip', ml)
            print 'maillist %s has duplicates by suid, skip' % ml
            continue
        yield ml


def iter_maillists_by_name(names):
    for n in names:
        if n=='__auto__':
            cache = set()
            for a in Autosubscription.objects.filter():
                if a.maillist.id not in cache:
                    cache.add(a.maillist.id)
                    yield a.maillist
        else:
            yield get_list(n)


def sync(options):

    if options.all_lists:
        maillists = iter_all_maillists(start_from=options.start_from_list)
    elif options.lists:
        maillists = iter_maillists_by_name(options.lists.split(','))
    else:
        print "--all-lists or --lists options required"
        sys.exit(1)

    for lst in maillists:
        print "--", lst.id, lst
        syncer = SyncSubs(maillist=lst, dry_run=options.dry_run)

        if DEBUG: print "sync imap"
        if not options.skip_imap:
            try:
                syncer.sync_imap_subscriptions()
            except YmailError as exc:
                print "ymail error", exc, "skip"

        if not options.skip_inbox:
            if not lst.email: continue
            try:
                syncer.sync_inbox_subscriptions()
            except CmailError as exc:
                print "cmail error", exc, "skip"
            except ValueError as exc:
                print "may be cmail error", exc, "skip"

class odict(dict):
    def __getattr__(self, name):
        return self[name]


OPTIONS = [

    {'args': ["", "--all-lists"],
     'kwargs': dict(dest="all_lists", action="store_true", help="Check all maillists")
    },

    {'args': ["", "--start-from-list"],
     'kwargs': dict(dest="start_from_list", default=None, help="If --all-list set, start from this list, not first")
    },

    {'args': ["", "--lists"],
     'kwargs': dict(dest="lists", default=None, help="Maillists to check (e.g. bbs,staff)"),
     },

    {'args': ["", "--dry-run"],
     'kwargs': dict(dest="dry_run", action="store_true", default=False, help="Dry run"),
     },

    {'args': ["", "--skip-inbox"],
     'kwargs': dict(dest="skip_inbox", action="store_true", default=False,
                    help="Do not check inbox (cmail) subscriptions")
    },

    {'args': ("", "--skip-imap"),
     'kwargs': dict( dest="skip_imap", action="store_true", default=False,
                     help="Do not check imap (mlapi) subscriptions"),
     }
]



class Command(BaseCommand):

    help = u"Синхронизация подписок с ymail и cmail"

    option_list = BaseCommand.option_list + tuple( [make_option(*o['args'], **o['kwargs']) for o in OPTIONS ] )

    def handle(self, *args, **options):
        return sync(odict(options))


if __name__=="__main__":
    from optparse import OptionParser
    parser = OptionParser()
    for o in OPTIONS:
        parser.add_option(*o['args'], **o['kwargs'])
    options, args = parser.parse_args()
    sync(options)