# -*- coding: utf-8 -*-

from django.db import models
from django.contrib.auth.models import User

from mlcore.permissions.utils import can_read


class MailListManager(models.Manager):

    def get_by_natural_key(self, name):
        return self.get(name=name)

    def filter_allowed_lists(self, user, lists):
        """
        @param user: User object or None
        @type user: django.contrib.auth.models.User

        @param lists: One MailList or iterable of MailLists
        @type lists: mixed

        @return: Allowed MailLists for current user
        @rtype: list of mailarhive.models.MailList
        """

        if not hasattr(lists, '__iter__'):
            lists = [lists]
        elif type(lists) not in (tuple, list, models.query.QuerySet):
            lists = list(lists)

        # detect open maillists
        open_lists = [ml for ml in lists if ml.is_open]

        # check user auth
        if not user or type(user) != User:
            return open_lists

        # detect closed maillists
        closed_lists = set(lists).difference(open_lists)
        allowed_closed_lists = (lst for lst in closed_lists if can_read(user, lst))

        open_lists.extend(allowed_closed_lists)
        return open_lists

    def exclude_deleted(self):
        return self.filter(is_deleted=False)
