# -*- coding: utf-8 -*-
import datetime
import re
import cPickle
from pprint import pformat

from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.utils.html import escape
from django.contrib.auth.models import User
from django.core.urlresolvers import reverse

from django_intranet_stuff.models import Staff
from django_intranet_stuff.utils.choices import Choices

from mlcore.utils.encoding import deutify_keys
from mlcore.subscribe.utils import load_backend_interface
from mlcore.subscribe import subscription_type
from mlcore.ml.operation_choices import OP_CHOICES_NAME, OP_CHOICES_TYPE
from mlcore.permissions.utils import can_write
from mlcore.permissions.models import SubscriptionRequest
from managers import MailListManager

import mlcore.permissions.signals_for_Log


DEFAULT_DOMAIN = '@yandex-team.ru'


def _user_display_name(self):
    if self.last_name:
        if self.first_name:
            return self.last_name + ' ' + self.first_name
        return self.last_name
    else:
        return self.username


def _user_intranet_profile(self, only=()):
    from django_intranet_stuff.models import Staff
    return Staff.objects.only(*only).get(login=self.username)


User.intranet_profile = _user_intranet_profile
User.display_name = _user_display_name
if not hasattr(Staff, 'get_full_name'):
    Staff.get_full_name = Staff.__unicode__
Staff.display_name = _user_display_name

# XXX: rude hack to export data with natural keys on user
User.natural_key = lambda self: (self.username,)
User.objects.get_by_natural_key = lambda username: User.objects.get(username=username)


ML_LANGUAGE = Choices(
    EN=('en', _('English')),
    RU=('ru', _('Russian')),
    TR=('tr', _('Turkish')),
)

ML_LANGUAGE.items = (c[0] for c in ML_LANGUAGE.choices())


ARCHIVING_TYPE = Choices(
    AR=('archive', _('Archiving')),
    CL=('clean', _('Cleaning')),
)
ARCHIVING_TYPE.items = (a[0] for a in ARCHIVING_TYPE.choices())


class MailList(models.Model):
    '''
    Maillist model.

    Models both imap and inbox maillists.
    '''

    ROOT_FOLDER = 'Yandex'

    name = models.CharField(max_length=255, db_index=True, unique=True)
    # alias is the name of maillist node in hierarchy, it may not be unique
    # for example, alias 'support.sm.video' is 'video'.
    # XXX: until we will gain control over maillist entities we SHOULD NOT
    # MODIFY ALIASES BY OUR OWN
    alias = models.CharField(max_length=255, db_index=True)
    email = models.EmailField(max_length=255, null=True, db_index=True)
    letter = models.CharField(max_length=1, db_index=True)
    info = models.TextField()
    info_en = models.TextField(default='')
    info_modified = models.BooleanField(default=False)
    max_length = models.PositiveIntegerField()

    fid = models.BigIntegerField(blank=True, null=True)
    fsuid = models.BigIntegerField(blank=True, null=True)

    # Flag properties
    # if is_imap is True, this maillist is created as shared folder and
    # imap subscription is allowed
    is_imap = models.BooleanField(default=False)
    # if is_sub is True, subscription by email or in user's inbox is allowed
    is_sub = models.BooleanField(default=False)
    # if maillist is marked as external (is_internal==False), anyone from
    # the outer world can send mails to this maillist
    is_internal = models.BooleanField(default=True)
    # внешние сотрудники по умолчанию не могут читать рассылку
    external_staff_can_read = models.BooleanField(default=True)
    # if maillist is open (is_open==True), anybody can subscribe, elsewhere
    # one need to ask owner for permission grant
    is_open = models.BooleanField(default=False)
    is_manual = models.BooleanField(default=False)
    is_sms = models.BooleanField(default=False)
    # if is_hidden is True, maillist will be hidden in the users page if
    # current user is not subscribed to the same maillist
    is_hidden = models.BooleanField(default=False)
    # if is_deleted is True, maillist will be hidden everywhere in the system,
    # but admin can unset this flag in the administrative interface
    is_deleted = models.BooleanField(default=False)
    # if is_mandatory, unsubscribe operations are restricted for users
    # (for example, staff@ is mandatory)
    is_mandatory = models.BooleanField(default=False)
    # if mailing list is virtual then it cannon have subscribers and
    # actually serves only as grouper for other lists to build
    # hierarchy ZOMBY ZOMBY
    is_virtual = models.BooleanField(default=False)
    readonly = models.BooleanField(default=False)

    # is_archive is True when owner wants autoremove old letters
    # after `before_clean` days. nice for cron letters (ML-971)
    is_archive = models.BooleanField(default=False)
    before_clean = models.IntegerField(null=True, blank=True, default=None)
    max_size = models.IntegerField(null=True, blank=True, default=None)
    archiving_type = models.CharField(max_length=7, default=ARCHIVING_TYPE.CL, choices=ARCHIVING_TYPE.choices())

    # if mailing list need to be tracked by mlcore.freshness, this flag
    # must be True.
    is_relevance_tracked = models.BooleanField(default=True)

    relevance = models.FloatField(db_index=True, default=0.0)

    preferred_language = models.CharField(max_length=6, db_index=True, default=ML_LANGUAGE.RU,
                                          choices=ML_LANGUAGE.choices())

    parent = models.ForeignKey('self', null=True, blank=True, related_name='children')

    imap_users = models.PositiveIntegerField(default=0)
    sub_users = models.PositiveIntegerField(default=0)

    created_dt = models.DateTimeField(null=True, db_index=True)
    created_at = models.DateTimeField(default=datetime.datetime.now)
    modified_at = models.DateTimeField(auto_now=True)

    use_cmail = models.BooleanField(default=False)

    objects = MailListManager()

    @property
    def info_firstline(self):
        return self.info.split('\n')[0] if self.info is not None else ''

    @property
    def info_firstline_en(self):
        return self.info_en.split('\n')[0] if self.info_en is not None else ''

    def get_alias_from_name(self):
        return self.name.rsplit('.', 1)[1]\
            if ('.' in self.name and '@' not in self.name)\
            else self.name

    def _get_normalized_name(self):
        """
        Normalized name for subscription.

        Appropriate to use in front-end templates.

        Examples:

        >>> m = MailList(name='tools')
        >>> m._get_normalized_name()
        u'tools'
        >>> m.name = 'support@service.yandex.ru'
        >>> m._get_normalized_name()
        u'support@service.yandex.ru'
        >>> m.name = 'svn_commits.wiki'
        >>> m._get_normalized_name()
        u'svn_commits.wiki'
        """
        norm = unicode(self.name)
        if ('.' not in norm and '@' not in norm and
                self.email and self.email.endswith(DEFAULT_DOMAIN)):
            norm += '@'

        return norm

    normalized_name = property(_get_normalized_name)

    def _get_exchange_name(self):
        '''
        Return maillist name converted for MS Exchange:

        >>> m = MailList(name='foo.bar-buz')
        >>> m._get_exchange_name()
        'yaFooBar-Buz'

        >>> m = MailList(name='foo.barbuz')
        >>> m._get_exchange_name()
        'yaFooBarbuz'

        >>> m = MailList(name='foobarbuz')
        >>> m._get_exchange_name()
        'yaFoobarbuz'

        '''

        def _capitalize(w):
            return '-'.join(s.capitalize() for s in w.split('-'))

        return 'ya' + ''.join(_capitalize(s) for s in self.name.split('.'))

    exchange_name = property(_get_exchange_name)

    def __unicode__(self):
        return self.normalized_name

    def subscribers_count(self):
        if not hasattr(self, "_subscribers_count"):
            self._subscribers_count = self.subscribers_set.filter(user__is_active=True).count() + self.emailsubscriber_set.count()
        return self._subscribers_count

    @property
    def owners(self):
        if not hasattr(self, "_owners"):
            self._owners = self.owner_set.all() if self.owner_set.count() > 0 else None
        return self._owners

    def save(self, *args, **kwargs):
        # self.set_as_modified()
        if not self.alias:
            self.alias = self.get_alias_from_name()
        super(MailList, self).save(*args, **kwargs)

    def natural_key(self):
        return (self.name,)

    def set_as_modified(self, at=None):
        """
        Обновляет время изменения текущей рассылки и всех других рассылок,
        на которые она может быть подписана.
        """
        at = at or datetime.datetime.now()

        if self.modified_at < at:
            # Получаем id всех дочерних рассылок
            maillist_ids = {self.id}

            maillists_to_lookup = list(self.get_child_maillists())
            while maillists_to_lookup:
                maillist = maillists_to_lookup.pop()

                if maillist.id in maillist_ids:
                    continue

                maillist_ids.add(maillist.id)
                maillists_to_lookup.extend(maillist.get_child_maillists())

            MailList.objects.filter(id__in=maillist_ids).update(modified_at=at)

    def get_child_maillists(self):
        if self.email:
            return MailList.objects.filter(emailsubscriber__email=self.email)

        return ()

    def get_email(self):
        if self.email:
            return self.email.lower()

        return '{0}@yandex-team.ru'.format(self.name.lower())

    @property
    def folder_path(self):
        maillists = [self.name]
        parent = self.parent
        while parent:
            # В целом не очень оптимально, но сильно вложенных рассылок у нас нет
            maillists.append(parent.name)
            parent = parent.parent

        maillists.append(self.ROOT_FOLDER)
        return list(reversed(maillists))

    class Meta:
        db_table = 'wakka_maillists'
        verbose_name = u'Рассылка'
        verbose_name_plural = u'Рассылки'
        ordering = ('name',)
        unique_together = [('alias', 'parent')]


class BackendContext(models.Model):
    '''
    Describes maillist context specific for backend.
    '''

    backend_type = models.CharField(max_length=32, db_index=True)
    maillist = models.ForeignKey(MailList, related_name='backends')

    # if is_imap is True, this maillist is created as shared folder and
    # imap subscription is allowed
    is_imap = models.BooleanField(blank=True, default=False)
    # if is_sub is True, subscription by email or in user's inbox is allowed
    is_sub = models.BooleanField(blank=True, default=False)

    @property
    def backend(self):
        if not hasattr(self, '_backend'):
            self._backend = load_backend_interface(self.backend_type)
        return self._backend

    @property
    def is_master_backend(self):
        return self.backend.is_master

    def get_clarified_context(self):
        return self.backend.model_class.objects.get(id=self.id)

    def as_dict(self):
        '''
        >>> m = MailList(id=100, name='bbs')
        >>> ctx = BackendContext(maillist=m, is_imap=True, backend_type='yandex_team')
        >>> ctx.as_dict()
        {'is_imap': True, 'backend_type': 'yandex_team', 'maillist_id': 100, 'is_sub': False}
        '''
        return {
            'backend_type': self.backend_type,
            'maillist_id': self.maillist_id,
            'is_imap': self.is_imap,
            'is_sub': self.is_sub,
        }

    @classmethod
    def from_dict(cls, ctx_as_dict):
        return cls(**deutify_keys(ctx_as_dict))

    class Meta:
        unique_together = [('backend_type', 'maillist')]


class Subscribers(models.Model):
    TYPE_DISPLAY_MAP = (
        (subscription_type.IMAP, _("shared folder")),
        (subscription_type.INBOX, _("inbox")),
        (subscription_type.BOTH, _("inbox, shared folder")),
        (subscription_type.NONE, _("no subscription")),
        (subscription_type.SEPARATE, _("separate folder"))
    )

    list = models.ForeignKey(MailList)
    user = models.ForeignKey(User, related_name='subscriptions', db_index=True)
    is_imap = models.BooleanField(default=False)
    is_sub = models.BooleanField(default=False)
    autosubscription = models.ForeignKey('Autosubscription', null=True, blank=True)
    # stype in u'both', u'imap', u'inbox', u'none'
    stype = models.CharField(max_length=15, db_index=True, choices=TYPE_DISPLAY_MAP)
    created_at = models.DateTimeField(null=False, auto_now_add=True, db_index=True)

    def __unicode__(self):
        if self.is_imap == True and self.is_sub == True:
            sub = 'imap and maillist'
        elif self.is_imap == True:
            sub = 'imap'
        else:
            sub = 'maillist'
        return self.user.username + ' subscribed on ' + self.list.name + ' by ' + sub

    def type_for_template(self):
        stype = dict(self.TYPE_DISPLAY_MAP).get(self.getType(), '')
        if self.list.readonly:
            write = _('write') if can_write(self.user, self.list) else ''
        else:
            write = ''
        return (stype, write)

    def getType(self):
        if self.is_imap and self.is_sub:
            return subscription_type.BOTH
        if self.is_imap:
            return subscription_type.IMAP
        if self.is_sub:
            return subscription_type.INBOX
        if not self.is_imap and not self.is_sub:
            return subscription_type.NONE
        raise ValueError("Unknown subscription type for id %d" % self.id)

    def get_access_reason(self):
        access_reason = None
        if not self.list.is_open:
            try:
                access_reason = SubscriptionRequest.objects.get(
                    permission__list=self.list, permission__user=self.user
                ).access_reason
            except SubscriptionRequest.DoesNotExist:
                access_reason = None
        return access_reason

    def save(self, *args, **kwargs):
        if not self.stype:
            self.stype = self.getType()
        # self.list.set_as_modified()
        super(Subscribers, self).save(*args, **kwargs)

    class Meta:
        db_table = 'wakka_subscribers'
        verbose_name = u'подписчик'
        verbose_name_plural = u'подписчики'


class EmailSubscriber(models.Model):
    list = models.ForeignKey(MailList)
    email = models.EmailField(max_length=255, db_index=True)
    created_at = models.DateTimeField(null=False, auto_now_add=True, db_index=True)

    def __unicode__(self):
        return "%s subscribed on %s" % (self.email, self.list.name)

    def type_for_template(self):
        return (u"входящие", )

    def getType(self):
        return subscription_type.EMAIL

    def save(self, *args, **kwargs):
        # self.list.set_as_modified()
        super(EmailSubscriber, self).save(*args, **kwargs)

    class Meta:
        db_table = 'mailarchive_emailsubscriber'
        verbose_name = u"email-подписчик"
        verbose_name_plural = u"email-подписчики"


class Owner(models.Model):
    user = models.ForeignKey(User)
    list = models.ForeignKey(MailList)
    created = models.DateTimeField(null=False, auto_now_add=True)

    def __unicode__(self):
        return "%s for %s@" % (self.user.username, self.list.name)

    class Meta:
        db_table = 'mailarchive_owner'
        unique_together = [('user', 'list')]
        verbose_name = u"ответственный"
        verbose_name_plural = u"ответственные"


class SuidLookup(models.Model):
    suid = models.BigIntegerField(primary_key=True)
    login = models.CharField(max_length=50, db_index=True)


class CorporateDomain(models.Model):
    MX_PDD = 'pdd'
    MX_CORP = 'corp'

    domain = models.CharField(max_length=100, unique=True)
    mx_type = models.CharField(max_length=8, choices=((MX_CORP, 'Корпоративный'), (MX_PDD, 'ПДД')), null=True)
    is_altdomain = models.BooleanField(default=False, help_text='Это домен типа=9 в yandex-team')
    comment = models.TextField(verbose_name=u"Комментарий", null=True)

    created_at = models.DateTimeField(null=True, auto_now_add=True)
    modified_at = models.DateTimeField(null=True, auto_now=True)

    @property
    def is_type9(self):
        return self.mx_type == self.MX_CORP and self.is_altdomain

    @property
    def is_pdd(self):
        return self.mx_type == self.MX_PDD


class LoginRedirect(models.Model):
    staff = models.ForeignKey(Staff, related_name='login_redirects')
    redirect_to = models.ForeignKey(Staff, related_name=None)

    class Meta:
        unique_together = (('staff', 'redirect_to'),)


class PendingBackendOperation(models.Model):
    '''
    Модель операции с рассылками, ожидающей выполнения.

    Позволяет сохранять информацию о неудавшихся операциях с почтовыми серверами
    и производить повторное выполнение по расписанию.
    '''
    DEFAULT_MAX_ATTEMPTS = 4
    FAILURE_INFO_LENGTH = 256

    created = models.DateTimeField(auto_now_add=True)
    is_notified = models.BooleanField(default=False)
    is_succeed = models.BooleanField(default=False)

    # pending/result
    attempts = models.PositiveSmallIntegerField(default=0)
    max_attempts = models.PositiveSmallIntegerField(default=DEFAULT_MAX_ATTEMPTS)
    last_failure_info = models.CharField(max_length=FAILURE_INFO_LENGTH, blank=True)

    # operation context (info needed to reperform it)
    backend_type = models.CharField(max_length=32)
    # operation id: +imap, -inbox, etc
    operation_type = models.CharField(max_length=32)
    # json-coded dict with arguments
    operation_args = models.CharField(max_length=256)

    class Meta:
        db_table = 'mailarchive_pendingbackendoperation'
        ordering = ('created',)
        verbose_name = u'Отложенная операция бэкенда'
        verbose_name_plural = u'Отложенные операции бэкенда'

    @property
    def totally_failed(self):
        '''
        True, если все попытки оказались неудачными
        '''
        return self.attempts >= self.max_attempts and not self.is_succeed

    def __unicode__(self):
        return u"%s %s, attempts %s/%s" % (self.backend_type,
                                           self.operation_type, self.attempts, self.max_attempts)


class PendingSubscribeOperation(PendingBackendOperation):
    '''
    Модель отложенной операции подписки/отписки.

    Позволяет хранить дополнительный контекст для операций подписки и отписки.
    Предок модели хранит только аргументы и id операции, поэтому нет возможности
    связать операцию с существующими списками и пользователями.
    '''
    list = models.ForeignKey(MailList, related_name='pending_subscriptions')
    user = models.ForeignKey(User, related_name='pending_subscriptions', db_index=True)

    class Meta:
        db_table = 'mailarchive_pendingsubscribeoperation'
        ordering = ('created',)
        verbose_name = u'Отложенная операция подписки'
        verbose_name_plural = u'Отложенные операции подписки'


class Autosubscription(models.Model):
    SUBSCRIPTION_TYPE = {
        subscription_type.IMAP: _("shared folder"),
        subscription_type.INBOX: _("inbox"),
        subscription_type.BOTH: _("inbox, shared folder")
    }
    maillist = models.ForeignKey(MailList, verbose_name='Maillist')
    group = models.ForeignKey('django_intranet_stuff.Group', verbose_name='Intranet Group')
    stype = models.CharField(max_length=15, choices=SUBSCRIPTION_TYPE.items(),
                             default=subscription_type.IMAP,
                             verbose_name=u'Subscription type')

    def __unicode__(self):
        return u'Autosubscription %s to %s' % (
            self.group.name,
            self.maillist.name
        )


class CreateListLock(models.Model):
    name = models.CharField(max_length=255)
    created_at = models.DateTimeField(auto_now_add=True, db_index=True)

    def __unicode__(self):
        return self.name


class OperationLog(models.Model):
    task_id = models.CharField(primary_key=True, max_length=255,
                               verbose_name=u"ID Операции")
    name = models.CharField(max_length=255, choices=OP_CHOICES_NAME,
                            verbose_name=u"Имя")
    type = models.CharField(max_length=255, choices=OP_CHOICES_TYPE,
                            verbose_name=u"Тип операции")
    started = models.DateTimeField(null=True, blank=True,
                                   verbose_name=u"Время начала", db_index=True)
    finished = models.DateTimeField(null=True, blank=True,
                                    verbose_name=u"Время завершения")
    parent = models.ForeignKey('self', null=True, blank=True, db_index=True,
                               related_name='children')
    root = models.ForeignKey('self', null=True, blank=True, db_index=True,
                             related_name='ifroot_descendants')
    comment = models.TextField(verbose_name=u"Комментарий")
    messages = models.TextField()
    user = models.ForeignKey(User, null=True, blank=True,
                             verbose_name=u"Пользователь")
    list = models.ForeignKey(MailList, null=True, blank=True,
                             verbose_name=u"Рассылка")
    status = models.CharField(max_length=100, db_index=True)
    initiator = models.CharField(max_length=255, db_index=True,
                                 verbose_name=u"Инициатор")
    arguments = models.TextField(default='')
    retry_parent = models.ForeignKey('self', null=True, blank=True,
                                     verbose_name=u"id таска который ретраили")
    comp_status = models.CharField(max_length=255, default='')

    def logs(self):
        if self.parent is not None:
            data = [self]
        else:
            data = self.root.ifroot_descendants.order_by('started')

        tmplt = '<br/><br/><pre><span id="ml-logs">%s</span></pre>'
        return tmplt % escape(
            '\n'.join('====== %s (%s)\n%s' %
                      (i.get_name_display(), i.task_id, i.messages)
                      for i in data))

    logs.short_description = u"Логи"
    logs.allow_tags = True

    def color_status(self, status):
        if '*' in status:
            tmplt = ('<span style="background-color: %s; padding:3px;"'
                     'title="Композитный статус">%s</span>')
        else:
            tmplt = '<span style="background-color: %s; padding:3px;">%s</span>'

        if 'OBSOLETE' in status:
            color = 'rgb(226, 226, 226)'
        elif 'FAILURE' in status:
            color = 'rgb(255, 174, 174)'
        elif 'RETRY' in status:
            color = 'rgb(255, 250, 139)'
        elif 'RETRIED' in status:
            color = 'rgb(255, 250, 139)'
        elif 'SUCCESS' in status:
            color = 'rgb(171, 255, 171)'
        else:
            return status
        return tmplt % (color, status)

    def comps_status(self, raw=False):
        """ Композитный статус для рутовых тасков """
        def comp():
            if self.parent is not None:
                return self.status

            statuses = set(self.root.ifroot_descendants.order_by('started')
                           .values_list('status', flat=True))
            if 'OBSOLETE' in statuses:
                return 'OBSOLETE*'
            elif 'FAILURE' in statuses:
                return 'FAILURE*'
            elif 'PENDING' in statuses:
                return 'PENDING*'
            elif all(map(lambda x: x == 'SUCCESS',
                         (i for i in statuses if i != 'RETRIED'))):
                return 'SUCCESS*'
            elif self.status == 'RETRIED':
                return 'RETRIED*'
            else:
                return '+'.join(i for i in statuses
                                if i not in ('SUCCESS', 'FAILURE', 'PENDING'))

        if not raw:
            return self.color_status(comp())
        else:
            return comp()

    comps_status.allow_tags = True
    comps_status.short_description = u'Общий статус'

    def link_name(self):

        if not self.task_id:
            # ML-1401 - ошибка, появилась вероятно при переезде на django 1.6
            # причину не нашел, поэтому делаю костыль
            return u"<span style='color: red;'>link_name:error</span>"

        url = reverse('admin:%s_%s_change' % (self._meta.app_label,
                                              self._meta.module_name),
                      args=[self.task_id])
        return u'<a href="%s">%s</a>' % (url, self.get_name_display())

    link_name.allow_tags = True
    link_name.short_description = u'Операция'

    def restart_button(self):
        if not 'FAILURE' in self.comps_status(raw=True):
            return ''

        tmplt = u"""<div class="manual-retry">
                        <input type="button" value="Перезапустить">
                        <input type="hidden" name="task_id" value="%(task_id)s"
                               data-url="%(url)s">
                    </div>"""
        data = {'url': reverse('retry_task'), 'task_id': self.task_id}
        return tmplt % data

    restart_button.allow_tags = True
    restart_button.short_description = u'Перезапустить'

    def pretty_arguments(self):
        if self.arguments:
            args = cPickle.loads(self.arguments.encode('ascii'))
            args.pop('context')
        else:
            return ''
        tmplt = """<pre>%s</pre>"""
        return tmplt % escape(pformat(args, width=30))

    pretty_arguments.allow_tags = True
    pretty_arguments.short_description = u'Аргументы'

    def short_id(self):
        return '-'.join(self.task_id.split('-')[:2]) + '-...'

    short_id.allow_tags = True
    short_id.short_description = u'ID операции'

    class Meta:
        ordering = ('-started', )
        verbose_name = u"Логи операции"
        verbose_name_plural = u'Логи операций'

    def __unicode__(self):
        return self.task_id


class DirectForwards(models.Model):
    STATUS_NEED_APPROVE = 'need_approve'
    STATUS_APPROVED = 'approved'
    STATUS_ACTIVE = 'active'
    STATUS_REVOKED = 'revoke'

    STATUS_CHOISES = {
        STATUS_NEED_APPROVE: _(u'ожидает подтверждения'),
        STATUS_APPROVED: _(u'подтверждена'),
        STATUS_ACTIVE: _(u'активна'),
        STATUS_REVOKED: _(u'отозвана'),
    }

    maillist = models.ForeignKey(MailList, verbose_name='Maillist')
    email = models.EmailField(max_length=255, db_index=True)
    status = models.CharField(max_length=16, choices=STATUS_CHOISES.items(),
                              verbose_name=u"Статус", default=STATUS_APPROVED)
    is_deleted = models.BooleanField(default=False)
    created_at = models.DateTimeField(null=False, auto_now_add=True, db_index=True)
    modified_at = models.DateTimeField(auto_now=True, editable=False)
    implemented = models.ForeignKey(EmailSubscriber, null=True, blank=True, db_index=True,
                                    related_name='direct_forwards')

    class Meta:
        unique_together = (("maillist", "email"),)

    @property
    def status_text(self):
        return DirectForwards.STATUS_CHOISES.get(self.status, self.status)
