from django.db import connection

from mlcore.permissions.utils import is_external_staff

_sql = """
# (maillists_id, uid, type)-relation
(
    # maillists with list permission
    SELECT `wakka_maillists`.`id` AS `maillist_id`,
            `wakka_maillists`.`name` AS `name`,
            `wakka_maillists`.`email` AS `email`,
           MIN(`yandex_team_context`.`uid`) AS `uid`,
           'has_direct_read_permission' AS `kind`
    FROM `wakka_maillists`
             INNER JOIN `permissions_listpermission`
                        ON (`wakka_maillists`.`id` = `permissions_listpermission`.`list_id`)
             INNER JOIN `permissions_type` ON (`permissions_listpermission`.`type_id` = `permissions_type`.`id`)
             LEFT OUTER JOIN `ml_backendcontext` ON (`wakka_maillists`.`id` = `ml_backendcontext`.`maillist_id`)
             LEFT OUTER JOIN `yandex_team_context`
                             ON (`ml_backendcontext`.`id` = `yandex_team_context`.`backendcontext_ptr_id`)
    WHERE (`permissions_listpermission`.`user_id` = %(user_id)s
           AND `permissions_type`.`name` = 'read'
           AND `permissions_listpermission`.`approved` = True
           AND NOT `wakka_maillists`.`is_deleted`
            {and_is_external_maillist}
            )
    GROUP BY `wakka_maillists`.`id`
)
UNION
(
    # subscriptions
    SELECT `wakka_subscribers`.`id`         AS `maillist_id`,
            `wakka_maillists`.`name` AS `name`,
            `wakka_maillists`.`email` AS `email`,    
           MIN(`yandex_team_context`.`uid`) AS `uid`,
           'subscribed_to'                   AS `kind`
    FROM `wakka_subscribers`
             LEFT OUTER JOIN `wakka_maillists` ON (`wakka_subscribers`.`list_id` = `wakka_maillists`.`id`)
             LEFT OUTER JOIN `ml_backendcontext` ON (`wakka_maillists`.`id` = `ml_backendcontext`.`maillist_id`)
             LEFT OUTER JOIN `yandex_team_context`
                             ON (`ml_backendcontext`.`id` = `yandex_team_context`.`backendcontext_ptr_id`)
    WHERE `wakka_subscribers`.`user_id` = %(user_id)s
    AND NOT `wakka_maillists`.`is_deleted`
    GROUP BY `wakka_subscribers`.`id`
)
UNION
(
    # owned maillists
    SELECT `wakka_maillists`.`id` AS `maillist_id`,
            `wakka_maillists`.`name` AS `name`,
            `wakka_maillists`.`email` AS `email`,            
           MIN(`yandex_team_context`.`uid`) AS `uid`,
           'owner_of' AS `kind`
    FROM `wakka_maillists`
             INNER JOIN `mailarchive_owner` ON (`wakka_maillists`.`id` = `mailarchive_owner`.`list_id`)
             LEFT OUTER JOIN `ml_backendcontext` ON (`wakka_maillists`.`id` = `ml_backendcontext`.`maillist_id`)
             LEFT OUTER JOIN `yandex_team_context`
                             ON (`ml_backendcontext`.`id` = `yandex_team_context`.`backendcontext_ptr_id`)
    WHERE `mailarchive_owner`.`user_id` = %(user_id)s
    AND NOT `wakka_maillists`.`is_deleted`
    {and_is_external_maillist}
    GROUP BY `wakka_maillists`.`id`
)
UNION
(
    # open maillists
    SELECT `wakka_maillists`.`id` AS `maillist_id`,
            `wakka_maillists`.`name` AS `name`,
            `wakka_maillists`.`email` AS `email`,
           MIN(`yandex_team_context`.`uid`) AS `uid`,
           'open' AS `kind`
    FROM `wakka_maillists`
             LEFT OUTER JOIN `ml_backendcontext` ON (`wakka_maillists`.`id` = `ml_backendcontext`.`maillist_id`)
             LEFT OUTER JOIN `yandex_team_context`
                             ON (`ml_backendcontext`.`id` = `yandex_team_context`.`backendcontext_ptr_id`)
    WHERE `wakka_maillists`.`is_open` = True
    AND NOT `wakka_maillists`.`is_deleted`
    {and_is_external_maillist}
    GROUP BY `wakka_maillists`.`id`
)
UNION
(
    # maillists with group permission
    SELECT `wakka_maillists`.`id`           AS `maillist_id`,
            `wakka_maillists`.`name` AS `name`,
            `wakka_maillists`.`email` AS `email`,
           MIN(`yandex_team_context`.`uid`) AS `uid`,
           'has_group_read_permission'      AS `kind`
    FROM `wakka_maillists`
             INNER JOIN `permissions_grouppermission`
                        ON (`wakka_maillists`.`id` = `permissions_grouppermission`.`list_id`)
             INNER JOIN `permissions_type` ON (`permissions_grouppermission`.`type_id` = `permissions_type`.`id`)
             LEFT OUTER JOIN `ml_backendcontext` ON (`wakka_maillists`.`id` = `ml_backendcontext`.`maillist_id`)
             LEFT OUTER JOIN `yandex_team_context`
                             ON (`ml_backendcontext`.`id` = `yandex_team_context`.`backendcontext_ptr_id`)
    WHERE `permissions_type`.`name` = 'read'
      AND `permissions_grouppermission`.`group_id` IN (
        # ancestors of all user groups
        SELECT `group`.`id`
        FROM `intranet_group` AS `group`
        WHERE `group`.`intranet_status` = 1
          AND `group`.`level` > 0
          # exists user group...
          AND EXISTS(
                SELECT `user_group`.`id`
                FROM (
                        # all groups of given user
                         SELECT DISTINCT `intranet_group`.`id`,
                                         `intranet_group`.`lft`,
                                         `intranet_group`.`rght`,
                                         `intranet_group`.`tree_id`
                         FROM `intranet_group`
                                  INNER JOIN `intranet_groupmembership`
                                             ON (`intranet_group`.`id` = `intranet_groupmembership`.`group_id`)
                         WHERE `intranet_groupmembership`.`staff_id` =
                               (SELECT `id` FROM `intranet_staff` WHERE `user_id` = %(user_id)s)
                     ) AS `user_group`
                # ...which is ancestor of group
                WHERE `group`.`lft` <= `user_group`.`lft`
                  AND `user_group`.`rght` <= `group`.`rght`
                  AND `group`.`tree_id` = `user_group`.`tree_id`
            )
    )
    AND NOT `wakka_maillists`.`is_deleted`
    {and_is_external_maillist}
    GROUP BY `wakka_maillists`.`id`
);
"""


_and_is_external_maillist = 'AND `wakka_maillists`.`external_staff_can_read`'


def get_permitted_maillists(user):
    """Return IDs and UIDs of maillist which user can read

    User can read maillist if:
        - user is already subscribed to maillist
        - maillist is open
        - user has direct read permission for maillist
        - user belongs to group which in turn has read permission for maillist
        - user is owner of maillist

    Returns:
        dictionary of form:
        >>> {
        >>>     'subscribed_to': [{'id': 1, 'uid': 2}],
        >>>     'owner_of': [{'id': 31, 'uid': 34534}],
        >>>     'has_direct_read_permission': [{'id': 133, 'uid': 345652}],
        >>>     'has_group_read_permission': [{'id': 123, 'uid': 785885}],
        >>>     'open': [{'id': 56, 'uid': 25654654}],
        >>> }
    """

    and_is_external_maillist = _and_is_external_maillist if is_external_staff(user) else ''
    query = _sql.format(user_id=user.id, and_is_external_maillist=and_is_external_maillist)

    cursor = connection.cursor()
    try:
        cursor.execute(query, params={'user_id': user.id})
        rows = cursor.fetchall()
    finally:
        cursor.close()

    maillists = {
        'subscribed_to': [],
        'owner_of': [],
        'has_direct_read_permission': [],
        'has_group_read_permission': [],
        'open': [],
    }
    for maillist_id, name, email, uid, kind in rows:
        maillists[kind].append({'id': maillist_id, 'uid': uid, 'name': name, 'email': email})

    return maillists
