# -*- coding: utf-8 -*-
import logging
from urllib import urlencode
import json

from django.conf import settings
from django.contrib.auth.models import User
from django.template import RequestContext
from django.http import HttpResponseRedirect, HttpResponse, Http404
from django.shortcuts import redirect, render_to_response, get_object_or_404, render
from django.utils.translation import ugettext as _
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_POST
from django.db import transaction
from django_intranet_stuff.models import Staff, Department

from mlcore.messager import Messager, AjaxMessager
from mlcore.ml.forms import FriendSubscribeForm, SearchForm, SubscribeMailList
from mlcore.ml.models import MailList, Owner, OperationLog, EmailSubscriber, DirectForwards
from mlcore.permissions.grant.user import is_user_owner, grant_user_permissions
from mlcore.permissions.models import ListPermission, SubscriptionRequest
from mlcore.permissions.revoke.user import deny_user_permissions
from mlcore.permissions.utils import can_read, is_external_staff
from mlcore.subscribe import subscription_type
from mlcore.tasks import retry_operations, operations, base
from mlcore.utils.getters import get_user
from mlcore.ml.search_list import list_searcher
from django.db.models import Q
from mlcore.interaction.yfurita import Autoreply, YFuritaError
from mlcore.ml.utils import user_has_autosubscription_maillist


@login_required
def feedback(request):
    return redirect('https://forms.yandex-team.ru/surveys/1283/')


def redirect_to_passport(request):
    retpath = request.META.get('HTTP_REFERER')
    if retpath is None:
        next_request = request.GET.get('next')
        if next_request:
            retpath = 'http%s://%s/%s' %\
                      ('s' if request.is_secure() else '',
                       request.get_host(), next_request.lstrip('/'))
        else:
            retpath = 'http%s://%s/' %\
                      ('s' if request.is_secure() else '',
                       request.get_host())
    login_url = settings.PASSPORT_LOGIN_URL + '&' + urlencode({'retpath': retpath})
    return HttpResponseRedirect(login_url)


def staff_redirect(request):
    if request.user.is_authenticated():
        return redirect("staff", username=request.user.username)
    else:
        return redirect("login")


def mlsearcher(request, mlname):
    lists = list_searcher(mlname)
    mlname = mlname.rstrip('@')

    data_lists = sorted(lists, key=lambda l: not l.name.startswith(mlname))

    if len(data_lists) == 1 and data_lists[0].name == mlname:
        return redirect('list', mlname)

    if not data_lists:
        messages.error(request, u'Похожих рассылок не нашлось =(')
        return staff_redirect(request)

    return render_to_response('mailarchive/search_result.html',
                                RequestContext(request, {
                                    'maillists': data_lists,
                                    'found_mname': mlname,
                                }))


@login_required
def search_maillist(request):
    form = SearchForm(data=request.REQUEST)
    if form.is_valid():
        mlname = form.cleaned_data['maillist']
        if not mlname:
            messages.error("Не найдено ни одной рассылки")
            return mlsearcher(request, mlname)
        mlname = mlname.strip(' \n\r\t,;')
        return mlsearcher(request, mlname)
    else:
        messages.error(request, _('You must specify at least one mailing list'))
        return staff_redirect(request)


@login_required
def subscription_requests(request):
    """
    List user's subscription requests and
    requests that they sent to user (if user is an owner of some maillists)
    """

    user = request.user

    my_requests = ListPermission.objects.filter(approved=False).filter(user=user)

    they_requests = ListPermission.objects.filter(approved=False).filter(
        list__in=Owner.objects.filter(user=user).values_list('list', flat=True)
    )

    return render_to_response('ml/requests.html', RequestContext(request, {
        'my_requests': my_requests,
        'they_requests': they_requests,
        'user': user,
    }))


@login_required
def discard_subscription_request(request, ml_name):
    """
    Delete subscription request
    """
    messager = Messager(request)
    try:
        maillist = MailList.objects.get(name=ml_name)
    except MailList.DoesNotExist:
        raise Http404

    operations.unsubscribe.delay({'initiator': request.user.username,
                                  'comment': u'Отказ от запроса доступа'},
                                 user=request.user.username,
                                 maillist=maillist.name)
    messager.success(
        _('Subscription request for mailling list %(listname)s will be delete.') %
        {'listname': maillist.name})

    return redirect('requests')


@login_required
@require_POST
def change_permissions(request, ml_name):
    is_ajax = request.is_ajax()

    if is_ajax:
        messager = AjaxMessager()
    else:
        messager = Messager(request)

    action = request.REQUEST["action"]
    users = request.REQUEST["user"].split(",")

    success = False

    try:
        maillist = MailList.objects.get(name=ml_name)

        if not is_user_owner(maillist, request.user):
            return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    except MailList.DoesNotExist:
        raise Http404

    perms = maillist.listpermission_set.filter(user__username__in=users, approved=False)

    if action == "allow":
        for perm in perms:
            try:
                req = perm.subscriptionrequest
            except SubscriptionRequest.DoesNotExist:
                operations.grant_permission.delay(
                    {'comment': u'Апрув доступа без подписки',
                     'initiator': request.user.username},
                    perm.user, perm.list, perm.type.name)
            else:
                operations.subscribe_user.delay(
                    {'comment': u'Апрув запроса на подписку',
                     'initiator': request.user.username},
                    perm.user, perm.list, req.type,
                    meddler=req.meddler, check_rights=False)
        success = True

    elif action == "deny":
        success = deny_user_permissions(perms, messager)
    elif action == "grant":
        success = grant_user_permissions(users, messager)
    else:
        messager.error('Bad request')

    if is_ajax:
        msg = u','.join([m[1] for m in messager.get_messages()])
        if success:
            response = HttpResponse(msg or 'OK', mimetype="text/plain", status=200)
        else:
            response = HttpResponse(msg or "FAIL", mimetype="text/plain", status=500)
    else:
        response = redirect('list', maillist.name)

    return response


@login_required
def subscribe_friend(request, ml_name):
    messager = Messager(request)
    form = FriendSubscribeForm(data=request.POST)

    if form.is_valid():
        user_list = User.objects.filter(username__in=form.cleaned_data['friends__login_ld'])
        access_reason = form.cleaned_data['access_reason']
        maillist = get_object_or_404(MailList, name=ml_name)
        # если хотя бы у одного указанного в поле друга нет доступов, то запрашиваем причину (см. ML-864)
        has_perms = ListPermission.objects.filter(user__in=user_list, list=maillist).count() == len(user_list)
        # если запрашивает подписку юзер, который является ответсвенным, то не спрашиваем причину запроса (см. ML-1290)
        owners = set(maillist.owner_set.values_list('user__username', flat=True))
        is_owner_request = True if request.user.username in owners else False
        # показывать запрос причины доступа нужно только один раз
        has_reason = form.cleaned_data['has_reason']

        # если рассылка закрытая и у коллег не было выдано доступа то нужно спрашивать причину
        need_request = not maillist.is_open and not has_perms
        # для ответсвенного не нужно показывать причину запроса
        need_request = False if is_owner_request else (need_request and not has_reason)
        if need_request:
            context = dict(list=maillist)
            context.update(form.cleaned_data)
            context.update(dict(has_reason=True))
            context.update(dict(form=form))
            return render(request, 'mailarchive/access_reason_for_friends.html', context)

        stype = form.cleaned_data['stype']

        for user in user_list:
            query_kwargs = dict(
                list=maillist
            )
            if stype == subscription_type.IMAP or stype == subscription_type.SEPARATE:
                query_kwargs.update(is_imap=True)
            elif stype == subscription_type.INBOX:
                query_kwargs.update(is_sub=True)
            elif stype == subscription_type.BOTH:
                query_kwargs.update(is_sub=True, is_imap=True)
            if user.subscriptions.filter(**query_kwargs).exists():
                messager.error(
                    _('%(username)s is already subscribed on the %(listname)s mailing list') %
                    {'username': user.username, 'listname': maillist.name})
                continue

            if not maillist.external_staff_can_read and is_external_staff(user):
                messager.error(
                    _(u'Can not subscribe %(username)s to this mailing list because they are an external user') %
                    {'username': str(user.username)})
                continue

            operations.subscribe_user.delay({
                    'initiator': request.user.username,
                    'comment': u'Подписка друга по причине: %s' % access_reason,
                },
                user, maillist, stype,
                meddler=request.user, access_reason=access_reason, send_notification=True
            )
            messager.info(_(u"Your friend %s will be subscribed") % user.username)
    else:
        messager.error(form.errors)
        messager.error(_('Choose at least one collegaue'))

    return redirect('list', ml_name)


def _run_subscribe_maillist(request, current_maillist, maillist):
    messager = Messager(request)
    es, created = EmailSubscriber.objects.get_or_create(email=maillist.email, list=current_maillist)
    if created:
        context = {
            'initiator': request.user.username,
            'comment': u'Подписка рассылки %s на рассылку %s' % (maillist.email, current_maillist.name)
        }
        base.subscribe_inbox(context, user=maillist.email, maillist=current_maillist.name).delay()
        messager.info(_(u"Maillist %(maillist)s will be subscribed") % {"maillist": maillist.name})
    else:
        messager.error(_(u"Maillist %(maillist)s is already subscribed") % {"maillist": maillist.name})
    return redirect('list', current_maillist.name)


def _has_circle(cur_maillist, maillist):
    """
    Проверяем, не получится ли цикла при подписики рассылки на рассылку
    """
    email_subscribers = cur_maillist.emailsubscriber_set.values_list('email', flat=True)
    # если нашли этот email в списке подписчиков, то не позволяем подписывать
    if maillist.email in email_subscribers:
        return True

    for e in email_subscribers:
        try:
            sub_maillist = MailList.objects.get(email=e)
            return _has_circle(sub_maillist, maillist)
        except MailList.DoesNotExist:
            pass
    return False


@login_required
def subscribe_maillist(request, ml_name):
    current_maillist = get_object_or_404(MailList, name=ml_name)
    maillist_name = request.POST['maillist__name']
    form = SubscribeMailList(data={'maillist': maillist_name})
    if form.is_valid():
        maillist_name = maillist_name.strip(' \n\r\t,;')
        maillist = MailList.objects.get(name=maillist_name)
        if not is_user_owner(maillist, request.user):
            messages.error(request, _('You can\'t subscribe this maillist. '
                                      'You should be owner by %(maillist)s.') % {'maillist': maillist.name})
            return redirect('list', ml_name)

        # ищем в current_maillist подписчиках рассылку maillist
        # ищем в подписчиках maillist рассылку current_maillist
        if ml_name == maillist_name or _has_circle(maillist, current_maillist) or _has_circle(current_maillist, maillist):
            messages.error(request, _('You can\'t subscribe this maillist. There will be a recursion'))
            return redirect('list', ml_name)
        return _run_subscribe_maillist(request, current_maillist, maillist)
    else:
        messages.error(request, _('You must specify at least one mailing list'))
        return redirect('list', ml_name)


@login_required
def change_subscription_type(request, ml_name, stype=None):
    messager = Messager(request)
    maillist = get_object_or_404(MailList, name=ml_name)
    if stype is None:
        stype = request.POST.get('stype')
    user = request.user

    # если запрос на подписку уже был создан, то смотрим на причину доступа (см. ML-864)
    access_reason = None
    has_perm = ListPermission.objects.filter(user=user, list=maillist)
    if not has_perm:
        subscription_request = SubscriptionRequest.objects.filter(permission__user=user, permission__list=maillist)
        access_reason = subscription_request[0].access_reason if subscription_request else request.POST.get('access_reason')

    if stype == subscription_type.NONE:
        if not user_has_autosubscription_maillist(user, maillist):
            operations.unsubscribe.delay({'initiator': user.username},
                                         user, maillist)
            messager.info(_(u"You will be unsubscribed in a few minutes"))
        else:
            messager.error(_(u"You can't unsubscribe from list of your department"))
    elif not user.subscriptions.filter(list=maillist).exists():

        # Если это внешний консультант, то проверяем дополнительные права
        if is_external_staff(user) and not maillist.external_staff_can_read:
            messager.info(_(u"You can’t subscribe to this mailing list because you are external user."))
        else:
            # Если рассылка закрытая и пользователю не был ранее выдан доступ и на предыдущем шаге не было указано
            # причины, то спрашиваем его, зачем ему подписываться? (см. ML-864) И если подписчик не ответственный
            # за рассылку (ML-1311)
            owners = set(maillist.owner_set.values_list('user__username', flat=True))
            if not maillist.is_open and not has_perm and not access_reason and user.username not in owners:
                return render(request, 'mailarchive/access_reason.html', {"list": maillist, "stype": stype})

            operations.subscribe_user.delay({'initiator': user.username},
                                            user, maillist, stype, access_reason=access_reason,
                                            send_notification=True)
            if can_read(user, maillist):
                messager.info(_(u"You will be subscribed in a few minutes"))
            else:
                messager.info(_(u"MSG.REQUEST_FOR_CLOSE_LIST_IS_SEND"
                                " %(list_name)s") % {'list_name': maillist.name})
    else:
        operations.change_subscription_type.delay({'initiator': user.username},
                                                  user, maillist, stype)
        messager.info(_(u"You will be subscribed in a few minutes"))

    return redirect('list', ml_name)


@login_required
def unsubscribe(request, ml_name):
    return change_subscription_type(request, ml_name, subscription_type.NONE)


def _get_list_by_id(id):
    # ищем список по email или по suid
    try:
        return MailList.objects.get(Q(email=id) | Q(name=id))
    except MailList.DoesNotExist:
        try:
            id = int(id)
            return MailList.objects.get(Q(fsuid=id) | Q(fid=id))
        except ValueError:
            pass
        except MailList.DoesNotExist:
            pass
    return None


@login_required
def unsubscribe_click(request, listid):
    maillist = _get_list_by_id(listid)
    if maillist:
        return change_subscription_type(request, maillist.name, subscription_type.NONE)
    else:
        # TODO: сделать юзер-френдли
        logging.warning('List %s not found', listid)
        return HttpResponse("No such list: %s" % listid, mimetype="text/plain", status=400)


@login_required
def mark_delete_ml(request, ml_name):
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    maillist.is_deleted = True
    maillist.save()
    # Обновляем кэш NWSMTP ML-1555
    maillist.set_as_modified()

    return redirect('list', ml_name)


@login_required
def delete_owner(request, ml_name, owner_name):
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    owner = get_object_or_404(Owner, list=maillist, user=get_user(owner_name))
    owner.delete()
    messages.info(request, _(u"Owner %(user)s was removed.") % {'user': owner})
    return redirect('list', ml_name)


@login_required
def delete_parent(request, ml_name):
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    operations.change_parent.delay(
        {
            'initiator': request.user.username,
            'comment': 'Parent will be deleted'
        },
        maillist,
        request.user,
        None
    )

    messages.info(request, _(u"Parent of %(maillist)s will be removed.") % {'maillist': maillist})
    return redirect('edit-other-list-properties', ml_name)


@login_required
def delete_autoreply(request, ml_name, a_id):
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)
    try:
        a = Autoreply()
        a.remove(maillist, a_id)
        messages.info(request, (u'Автоответ удален.'))
    except YFuritaError as err:
        messages.error(request, (u'Сбой при удалении автоответа. Попробуйте позже'))
    return redirect('autoreply', ml_name)


@login_required
def set_priority_autoreply(request, ml_name, main_id):
    """
    Автоответ по заданному id становится наиболее приоритетным из всех остальных.
    """
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    a = Autoreply()
    all_ids = [r['id'] for r in a.detailed(maillist)]
    all_ids.pop(main_id)
    all_ids.insert(0, main_id)
    a.remove(maillist, all_ids)
    return redirect('autoreply', ml_name)


@login_required
def set_main_autoreply(request, ml_name, main_id):
    """
    Автоответ по заданному id остается включенным, а все остальные - выключаются.
    """
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)
    try:
        a = Autoreply()
        all_ids = [r['id'] for r in a.detailed(maillist)]
        all_ids.remove(main_id)
        for id_rule in all_ids:
            a.enable(maillist, id_rule, 0)  # 0 -> выключаем правило
        a.enable(maillist, main_id, 1)  # 1 -> включаем главное правило
        messages.info(request, (u'Установка параметров вкл./выкл.'))
    except YFuritaError as err:
        messages.error(request, (u'Сбой при установке параметров. Попробуйте позже, %s'%err))
    return redirect('autoreply', ml_name)


@login_required
def set_enabled_autoreply(request, ml_name, rule_id, enable):
    """
    Автоответ по заданному id либо включается, либо выключается.
    """
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)
    try:
        a = Autoreply()
        enabled = 1 if bool(int(enable)) else 0
        a.enable(maillist, rule_id, enabled)
        messages.info(request, (u'Установка параметров вкл./выкл.'))
    except YFuritaError as err:
        messages.error(request, (u'Сбой при установке параметров. Попробуйте позже, %s'%err))
    return redirect('autoreply', ml_name)


@login_required
def change_priorities_autoreply(request, ml_name, rule_ids):
    """
    Изменяем приоритеты всех правил
    """
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)
    try:
        a = Autoreply()
        # 1. меняем разделитель id-шников, как в furita-api
        # 2. делаем реверс приоритетов, так как на странице редактирования первое правило
        # считается самом приоритетным, а в API первое правило в списке считается менее приоритетным
        ids = ','.join(rule_ids.split('_')[::-1])
        a.order(maillist, ids)
        messages.info(request, (u'Изменение приоритетов'))
    except YFuritaError as err:
        messages.error(request, (u'Сбой при установке параметров. Попробуйте позже, %s'%err))
    return redirect('autoreply', ml_name)


@login_required
def info_delete_ml(request, ml_name):
    maillist = get_object_or_404(MailList, name=ml_name)
    if not is_user_owner(maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    return render(request, 'mailarchive/show_del_info.html', {"list": maillist})


def index_maillists(request):
    """ Индекс всех рассылок с родителями и признаком открытая или закрытая: ML-1078 """
    maillist = MailList.objects.all()[0]
    return render(request, 'mailarchive/index_maillists.html', {'list': maillist})


def index_gosmaillists(request):
    """ Индекс гос. россылок из таска: ML-1097 """
    exclude_list = [name+'@staff.yandex-team.com' for name in ('monitor', 'moscow', 'everyone')]
    maillists = MailList.objects.exclude_deleted().exclude(name__in=exclude_list).filter(
        Q(name__endswith='@staff.yandex-team.com') |
        Q(name__exact='staff') |
        Q(name__exact='redrose-announces')).order_by('name')

    return render(request, 'mailarchive/index_gosmaillists.html', {'lists': maillists})


@transaction.non_atomic_requests('default')
@transaction.non_atomic_requests('slave')
def ping(request):
    return HttpResponse('OK', status=200)


def retry_task(request):
    """ Перезапускает таску """
    task = get_object_or_404(OperationLog, task_id=request.POST.get('task_id'))
    retry_operations([task], initiator=request.user.username)
    messages.info(request, (u'Операция "%s" перезапущена. '
                            u'Обновите страницу') % task.get_name_display())
    return redirect(request.META.get('HTTP_REFERER', '/'))


@login_required
def subscribers_lookup(request):
    subscribers_result = []
    if request.method == "GET":
        maillist = get_object_or_404(MailList, name="testml")
        if u'query' in request.GET:
            value = request.GET[u'query']
            user_subscriptions_base = maillist.subscribers_set.filter(user__is_active=1)
            user_subscriptions = user_subscriptions_base.select_related("user__staff").order_by(
                "user__last_name",
                "user__first_name"
            ).filter(user__email_icontains=value).all()
            email_subscriptions = maillist.emailsubscriber_set.filter(email__icontains=value).all()
            subscribers_result = list(user_subscriptions) + list(email_subscriptions)
    json_res = json.dumps(subscribers_result)
    return HttpResponse(json_res, mimetype='application/json')


@login_required
def get_department_staff(request):
    department_id = request.GET.get('department_id')
    department = get_object_or_404(Department, pk=department_id)
    is_big_boss = request.GET.get('is_boss') == 'true'

    staffs = Staff.objects.filter(department__in=department.get_descendants(include_self=True), is_dismissed=False)
    if is_big_boss:
        staffs = staffs.filter(is_big_boss=True)

    json_res = json.dumps(
            [dict(login=staff.login, login_ld=staff.login_ld, name=staff.get_full_name()) for staff in staffs]
    )
    return HttpResponse(json_res, mimetype='application/json')


@login_required
def delete_forward(request, ml_name, forward_id):
    fwd = get_object_or_404(DirectForwards, id=forward_id, maillist__name=ml_name)
    if not is_user_owner(fwd.maillist, request.user):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    task_context = {
        'initiator': request.user.username,
        'comment': 'Delete direct forward for %s' % fwd.email,
    }
    base.delete_direct_forward(task_context, fwd.id)
    return redirect('forward', ml_name)
