# encoding: utf-8

from django.http import HttpResponse, Http404
from django.db.models import Q
import json
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import User

from django_api_auth.decorators import passport_or_token_auth
from mlcore.ml.models import MailList
from mlcore.ml.search_list import list_searcher
from mlcore.permissions import permission


@passport_or_token_auth
def external_lists(request):
    string = '\n'.join(
        MailList.objects.exclude_deleted().filter(is_internal=0)
            .values_list('email', flat=True))
    return HttpResponse(string, mimetype="text/plain")


@passport_or_token_auth
def non_project_lists(request):
    string = '\n'.join(
        MailList.objects.exclude_deleted().exclude(servergroup='project')\
            .exclude(email='').exclude(email__isnull=True)\
            .values_list('email', flat=True))
    return HttpResponse(string, mimetype="text/plain")


LIST_INFO_VALID_FIELDS = set( ['name', 'is_open', 'email', 'info', 'exchange_name'] )
LIST_INFO_DEFAULT_FIELDS = ( 'name', 'is_open', 'email', 'info' )

@passport_or_token_auth
def lists_info(request):
    """
    Cписок всех рассылок с описанием.
    include_unsub – параметр который позволяет запросить все рассылки
        в т.ч. на которые нельзя подписаться
    fields - параметр позволяет выбрать определённые поля, которые будут в выдаче
    """

    include_unsub = request.REQUEST.get('include_unsub', False)
    data = MailList.objects.exclude_deleted()

    # Удалим из выдачи скрытые рассылки ML-1303
    data = data.filter(is_hidden=False)

    fields = request.REQUEST.get('fields', None)
    if fields:
        fields = list( LIST_INFO_VALID_FIELDS.intersection( fields.split(',') ) )

    if not fields:
        fields = LIST_INFO_DEFAULT_FIELDS

    if not include_unsub:
        data = data.exclude(is_sub=False, is_imap=True)

    # was: maillists = list(data.values( *fields ))
    # new version is slower, but can handle 'exchange_name' property

    maillists = []
    for d in data:
        row = {}
        for k in fields:
            row[k] = getattr(d, k, None)  
        maillists.append( row  )
    
    string = json.dumps({
        'maillists': maillists
    })
    return HttpResponse(string, mimetype="text/plain")


def list_names(request):
    return HttpResponse(u'\n'.join(MailList.objects.exclude_deleted().values_list('name', flat=True)),
            mimetype='text/plain')


@login_required
def lists_by_letter(request, letter):

    letter = letter.upper()

    subscribes = request.user.subscriptions\
                 .filter(Q(is_imap=1) | Q(is_sub=1)).values_list('list__id', flat=True)\
                if request.user.is_authenticated() else ()

    permissions = permission.get_all_list_permissions(request.user)

    items = {}
    tree = []

    for mldict in (MailList.objects.exclude_deleted()
                .filter(letter=letter)
                .order_by('name')
                .values('id', 'name', 'letter', 'is_imap', 'is_sub',
                        'is_open', 'is_internal', 'is_hidden', 'relevance', 'parent')):

        mldict['is_subscribed'] = mldict['is_access'] = mldict['id'] in subscribes
        if not mldict['is_access']:
            mldict['is_access'] = mldict['name'] + '.read' in permissions

        name = mldict['name'] = mldict['name'].strip('.')

        if ('.' in name and '@' not in name) or mldict["parent"] is not None:
            if '.' in name and '@' not in name:
                parent, title = name.rsplit('.', 1)
            elif mldict["parent"]:
                parent = MailList.objects.get(id=mldict["parent"]).alias
                title = name

            mldict['title'] = title
            if parent not in items:
                items[parent] = {'title': parent, 'name': parent}
            parent_dict = items[parent]

            if not parent_dict.has_key('children'):
                parent_dict['children'] = []

            old_item = items.get(name, {})
            old_item.update(mldict)
            items[name] = old_item

            parent_dict['children'].append(old_item)

        else:
            title = name
            mldict['title'] = title
            old_item = items.get(name, {})
            old_item.update(mldict)
            tree.append(old_item)
            items[name] = old_item

    data = json.dumps(tree)
    return HttpResponse(data, mimetype="text/plain")


@passport_or_token_auth
def lists_by_owner(request, login):
    """
    Список логинов ответственных за рассылку
    """
    try:
        user = User.objects.get(username=login)
    except User.DoesNotExist:
        raise Http404

    data = user.owner_set.values_list('list__name', flat=True)

    string = json.dumps({
        'maillists': list(data)
    })

    return HttpResponse(string)


@login_required
def lists_by_partname(request):
    """
    Выдаем список рассылок, содержащих запрашиваемое название.
    Если название не задано, то выдаем error.
    """

    name_part = request.REQUEST.get('q', '')
    lists = list_searcher(name_part)
    if not lists:
        return HttpResponse(json.dumps({'error': 'no find maillist'}), content_type='application/json')

    items = {}
    tree = []

    data_lists = lists.values('id', 'name', 'letter', 'is_imap', 'is_sub',
                        'is_open', 'is_internal', 'is_hidden', 'relevance', 'parent')
    data_lists = sorted(data_lists, key=lambda d: not d['name'].startswith(name_part) )

    for mldict in data_lists:
        name = mldict['name'] = mldict['name'].strip('.')

        if ('.' in name and '@' not in name) or mldict["parent"] is not None:
            if '.' in name and '@' not in name:
                parent, title = name.rsplit('.', 1)
            elif mldict['parent']:
                parent = MailList.objects.get(id=mldict['parent']).alias
                title = name

            mldict['title'] = title
            if parent not in items:
                items[parent] = {'title': parent, 'name': parent}
            parent_dict = items[parent]

            if not parent_dict.has_key('children'):
                parent_dict['children'] = []

            old_item = items.get(name, {})
            old_item.update(mldict)
            items[name] = old_item

            parent_dict['children'].append(old_item)

        else:
            title = name
            mldict['title'] = title
            old_item = items.get(name, {})
            old_item.update(mldict)
            tree.append(old_item)
            items[name] = old_item

    data = json.dumps(tree)
    return HttpResponse(data, mimetype="text/plain")
