# encoding: utf-8
import logging
from mlcore.subscribe import subscription_type
from mlcore.ml.models import LoginRedirect
from django_intranet_stuff.models import Staff
from django.conf import settings
from .utils import EXCHANGE_DOMAIN, MAIL_DOMAIN, YT_DOMAIN, DEVNULL_EMAIL, normalize_email
from .utils import get_staff_by_email, get_maillist_by_email, is_email, get_shared_folder_email
from mlcore.pg_migration.utils import get_maillist_replica_email, get_maillist_replica_uid, PGMigrationError

class Expander(object):
    @staticmethod
    def _expand_staff(staff):
        redirects = LoginRedirect.objects.filter(staff=staff) \
            .values_list('redirect_to', flat=True).all()

        if redirects:
            # Если есть редиректы - отдаём только редиректы
            for redirected_staff_id in redirects:
                s = Staff.objects.get(id=redirected_staff_id)
                for e in Expander._expand_staff(s):
                    yield e

        elif staff.is_dismissed:
            # Если уволенный сотрудник, то ничего не отдаем
            return

        elif staff.has_exchange:
            # Если основной ящик в exchange - отдаём ящик exchange
            # И еще отдаем ящик в домене mail.yandex-team.ru (см. ML-1454)
            yield staff.login + EXCHANGE_DOMAIN
            yield staff.login + MAIL_DOMAIN

        elif staff.affiliation == 'yamoney':
            # Сотрудник яндекс-денег
            if staff.work_email:
                yield staff.work_email
                yield staff.login + MAIL_DOMAIN

        else:
            yield staff.login + MAIL_DOMAIN

    @staticmethod
    def _expand_maillist(maillist, cache):
        if cache is None:
            cache = set()

        # 0. Получаем imap-email для рассылки
        my_shared_folder_email = get_shared_folder_email(maillist)

        if my_shared_folder_email in cache:
            return

        if my_shared_folder_email:
            yield my_shared_folder_email

        # TODO: выпилить после переезда в PG
        if settings.PG_MIGRATION_STATE != settings.PG_MIGRATION_STATE_PG:
            try:
                # Если есть реплика в BB
                if get_maillist_replica_uid(maillist, raise_on_error=False):
                    yield get_maillist_replica_email(maillist)
            except PGMigrationError:
                logging.exception('Can\'t resolve PG replica')

        # Если рассылка удаленная, то возвращаем только shared_folder,
        # а подписчиков во входящих не отдаем (см. ML-1467)
        if maillist.is_deleted:
            return

        cache.add(my_shared_folder_email)
        # 1. Сначала пользователи-подписчики
        # Здесь не может быть рассылок и не может быть дублей,
        # но могут быть пользовательские алиасы и сотрудники
        for email in maillist.subscribers_set.filter(
                stype__in=[subscription_type.BOTH, subscription_type.INBOX, subscription_type.SEPARATE]
        ).values_list('user__email', flat=True):
            for _ in Expander.expand(email=email, cache=cache):
                yield _

        cache.add(maillist.email)
        # 2. Потом "внешние" подписчики.
        # Здесь могут быть другие рассылки, внешние email-ы и алиасы
        for email in maillist.emailsubscriber_set.all().values_list('email', flat=True):
            email = normalize_email(email)
            if email in cache:
                continue
            for _ in Expander.expand(email=email, cache=cache):
                yield _

    @staticmethod
    def expand(email, cache=None):
        if not cache:
            cache = set()

        if not is_email(email):
            return

        # 0. Если devnull
        if email == 'devnull' or email.startswith('devnull@'):
            yield DEVNULL_EMAIL

        maillist = get_maillist_by_email(email)
        staff = get_staff_by_email(email)

        # 1. Если рассылка
        if maillist:
            for _ in Expander._expand_maillist(maillist, cache):
                yield _
        # 2. Если пользователь стаффа
        elif staff:
            for _ in Expander._expand_staff(staff):
                yield _
        # 3. Если доменный email
        elif not email.endswith(YT_DOMAIN) and email != 'devnull':
            yield email
