# encoding: utf-8
from functools import wraps

from django.core.management.base import BaseCommand
from django.core.exceptions import ObjectDoesNotExist, MultipleObjectsReturned
from mlcore.ml.models import MailList
from mlcore.nwsmtp_connect.views import FORMAT_VERSION, force_update_cache, normalize_email
from mlcore.nwsmtp_connect.models import NwsmtpInfoCache
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext
from optparse import make_option


import logging
logger = logging.getLogger(__name__)

YT_DOMAIN = '@yandex-team.ru'


def get_all_maillists():

    def _get_email(maillist):
        return normalize_email(maillist.email)

    def _get_imapemail(maillist):
        try:
            context = YandexTeamBackendContext.objects.get(maillist=maillist)
            return context.passport_name+'@yandex-team.ru'
        except YandexTeamBackendContext.DoesNotExist:
            logging.info("YandexTeamBackendContext not found for %s" % maillist.email)

    for maillist in MailList.objects.filter(is_deleted=False):
        if not maillist.email:
            continue

        email = _get_email(maillist)
        # Обновляем информацию по рассылкам для нормальных default-email-ов
        if email:
            yield email, maillist.modified_at
        # Обновляем информацию по рассылкам для imap_email-ов в домене @yandex-team.ru
        # (support@yandex.money.ru -> support-at-yandex-at-money@yandex-team.ru)
        if not maillist.email.endswith(YT_DOMAIN):
            email = _get_imapemail(maillist)
            if email:
                yield email, maillist.modified_at


class Command(BaseCommand):
    help = u"Обновление кэша для nwsmtp"

    option_list = BaseCommand.option_list + (
        make_option(
            '--force',
            action='store_true',
            default=False,
            help='Force update cache'
        ),
        make_option(
            '-m',
            '--maillists',
            dest='input_maillists',
            default='',
            help='For input maillists'
        ),
    )

    def handle(self, *args, **options):
        if options.get('input_maillists'):
            emails = [email.strip() for email in options.get('input_maillists').split(',')]
            for email in emails:
                force_update_cache(email)
            return

        for email, modified_at in get_all_maillists():
            try:
                cached = NwsmtpInfoCache.objects.get(email=email)
            except ObjectDoesNotExist:
                # Обновить, если кэша нет совсем
                cached = None

            if cached and cached.version != FORMAT_VERSION:
                # Обновить, если старая версия формата
                cached = None

            if cached and cached.modified_at < modified_at:
                # Обновить, если список более свежий, чем кэш
                cached = None

            if options.get('force'):
                cached = None

            if not cached:
                try:
                    force_update_cache(email)
                except MultipleObjectsReturned:
                    logging.exception('Can not update nwsmtp cache for %s', email)
