# encoding: utf-8
import logging
from mlcore.ml.models import MailList
from mlcore.nwsmtp_connect.models import NwsmtpInfoCache
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext
from django_intranet_stuff.models import Staff
from mlcore.utils.blackbox_util import get_all_emails_by_uid, get_uid_by_email, get_userinfo
import blackbox

EXCHANGE_DOMAIN = '@ld.yandex.ru'
MAIL_DOMAIN = '@mail.yandex-team.ru'
YT_DOMAIN = '@yandex-team.ru'
DEVNULL_EMAIL = 'devnull@mail.yandex-team.ru'
YANDEX_TEAM_DOMAINS = ['yandex-team.ru', 'yandex-team.com', 'yandex-team.com.ua', 'yandex-team.com.tr']

# Костыль для ML-1443 (из-за того, что паспорт считает точки и тире одним и тем же)
EXCEPT_SOME_EMAILS_WITH_POINTS = ['a.pavlova@yandex-team.ru', 'd.melnikova@yandex-team.ru', 'pavlova.m@yandex-team.ru']
EXCEPT_SOME_EMAILS_WITH_DASH = ['a-pavlova@yandex-team.ru', 'd-melnikova@yandex-team.ru', 'pavlova-m@yandex-team.ru']

try:
    TEST_EMAILS = dict([(e, e.split('@')[0]+'@yandex-team.ru') for e in
                        open('/etc/ml-nwsmtp-test-email-substitute.txt').read().split()])
    logging.info('read nwsmtp substitute emails: %s', TEST_EMAILS)
except IOError:
    TEST_EMAILS = {}


def normalize_email(raw_email):
    # точки и тире не нормализуем, т.к. могут придти и в таком кривом виде
    email = raw_email and raw_email.strip().lower() or raw_email
    # к нам приходит рассылка в домене mail.yandex-team.ru, и мы её различаем с yandex-team.ru
    # if email:
    #     email = email.replace(MAIL_DOMAIN, YT_DOMAIN)
    return email


def get_imap_name_domain(email):
    name, domain = email.split('@')
    # разворачиваем imap-email-ы (названия из паспорта) в домене yandex-team.ru
    if email.endswith(YT_DOMAIN):
        imap_name = name.replace('.', '-')
    else:
        domain_suffix = domain.replace('.', '-')
        imap_name = u'-at-'.join([name.replace('.', '-'), domain_suffix])
    return imap_name, domain


def get_maillist_by_imap_email(email):
    # Обрабатываем email-ы вида: support-at-yandex-money-ru@yandex-team.ru в yandex-team.ru-домене
    if not email.endswith(YT_DOMAIN):
        return None

    imap_name, _ = email.split('@')
    # есть случаи, когда удаленная и актуальная рассылки имеют одинаковые imap_name-имена
    # например рассылки с точками (partner-support@market.yandex.ru & partner.support@market.yandex.ru)
    ytbacks = YandexTeamBackendContext.objects.filter(
        passport_name=imap_name,
    ).order_by('maillist__is_deleted')
    # в этом случае возвращаем не удаленную рассылку, а актуальную
    # является устаревшим поведением для некоторых рассылок, в будущем такого не должно быть
    # из интерфейса невозможно создать
    # только если руками кто-то в базе прописал
    if ytbacks:
        return ytbacks[0].maillist
    return None


def get_maillist_by_email_with_tld_domain(email):
    # Обрабатываем email-ы вида: exist_name@yandex-team.ru|com|com.ua|com.tr
    # ML-1472 и ML-1473:
    # с одной стороны в корп. почте пользователь имеет алиасы
    # yandex-team.ru | com | com.tr | com.ua
    # с другой стороны на корпе имеются различающиеся рассылки вида
    # login@yandex-team.com и login@yandex-team.com.tr
    # Предполагаемое решение когда nwsmtp приходил в ML искать email вида
    # login@yandex-team.com
    # нужно:
    #       1. производить поиск в базе по именно этому email, в случае успеха это будет означать, что
    #       найдена рассылка-исключение и она отличается от рассылки с тем же login,
    #       но в домене yandex-team.ru
    #       2. иначе повторно искать и разворачивать рассылку уже в домене
    #       yandex-team.ru, подразумевая, что эта рассылка имеет алиасы во всех корпоративных доменах
    name, domain = email.split('@')
    if domain in YANDEX_TEAM_DOMAINS:
        # тут должна вернуться рассылка в домене yandex-team.ru
        maillists = MailList.objects.filter(name=name)
        if maillists:
            return maillists[0]
    # если на предыдущих шагах не нашлась рассылка в доменах, отличных от YANDEX_TEAM_DOMAINS,
    # то значит, такой рассылки нет
    return None


def get_cache_by_email_tld_domain(email):
    # По email-e вида hello@yandex-team.com получаем кеш для существующей рассылки в домене yandex-team.ru
    # (т.е. кеш для hello@yandex-team.ru)
    maillist = get_maillist_by_email_with_tld_domain(email)
    if not maillist:
        return None

    cached = NwsmtpInfoCache.objects.filter(email=maillist.email)
    if not cached:
        return None
    return cached[0]


def get_cache_by_email(email):
    if not is_email(email):
        return None
    cached = NwsmtpInfoCache.objects.filter(email=email)
    if not cached:
        return get_cache_by_email_tld_domain(email)
    return cached[0]


def check_email_is_yt(email):
    name, domain = email.split('@', 1)
    return domain in YANDEX_TEAM_DOMAINS


def get_maillist_by_email(email):
    if not email or email.count('@') != 1:
        return None
    if email in EXCEPT_SOME_EMAILS_WITH_DASH:
        logging.warning('conflict user email and maillist email, see more: ML-1443')
        return None

    try:
        # не должно быть одинаковых email-ов у рассылки
        # TODO: придумать и понять, в каком месте делать проверку на уникальность email-ов.
        maillist = MailList.objects.get(email=email)
    except MailList.DoesNotExist:
        # возможно пришел email вида passport_name@yandex-team.ru?
        maillist = get_maillist_by_imap_email(email)
    if not maillist:
        # может пришел email с существующим названием рассылки, но в другом домене?
        return get_maillist_by_email_with_tld_domain(email)
    return maillist


def get_shared_folder_email(maillist):
    context = YandexTeamBackendContext.objects.filter(maillist=maillist).\
        values_list('passport_name', flat=True)
    if maillist.email in EXCEPT_SOME_EMAILS_WITH_POINTS:
        logging.warning('conflict user email and maillist email, see more: ML-1443')
        return None
    if not context:
        logging.warning('YandexTeamBackendContext not found for %s', maillist)
        return None
    if len(context) > 1:
        logging.warning('More than 1 YandexTeamBackendContext for %s. Get first', maillist)
    return context[0] + MAIL_DOMAIN


def get_staff_by_email(email, strict=False):
    # на этом этапе проверяем домен, т.к. паспорт на
    # @mail.yandex-team.ru|com|com.tr|com.ua выдает то же, что и для @yandex-team.ru
    email_suffixes = ['@mail.yandex-team.ru', '@mail.yandex-team.com', '@mail.yandex-team.com.tr',
                      '@mail.yandex-team.com.ua', '@ld.yandex.ru']
    for suff in email_suffixes:
        if email.endswith(suff):
            return None

    # email проверяем через паспорт, потому что модель Staff не знает все email-адреса
    # TODO: кэшировать ответы паспорта, чтобы ускорить ответ
    ui = get_userinfo(email, is_yateam=True, by_login=True)
    login = ui['fields'].get('uid')
    if login:
        try:
            return Staff.objects.get(login=login)
        except Staff.DoesNotExist:
            logging.warning('User %s found by %s does not exist in staff', login, email)
            if strict:
                raise


def is_email(email):
    if not email:
        return False
    if email == 'devnull':
        return True
    return email.count('@') == 1


def get_passport_aliases(emails):
    if isinstance(emails, str):
        emails = [emails]
    elif isinstance(emails, unicode):
        emails = [emails]

    aliases = []
    for email in emails:
        try:
            uid = get_uid_by_email(email)
            if uid:
                uid_emails = get_all_emails_by_uid(uid)
                aliases.extend(uid_emails)
        except blackbox.BlackboxError as err:
            logging.error('get_passport_aliases failed for %s: %s', email, err)
    return aliases

