# coding: utf-8
import json
import logging

from django.http import HttpResponse
from django.views.decorators.http import require_GET
from django.db import transaction, DatabaseError

from ..failover_router import prefer_slave
from .models import NwsmtpInfoCache
from .utils import normalize_email, get_cache_by_email, is_email, check_email_is_yt
from .emailinfo import EmailInfo


FORMAT_VERSION = '1.0'

logger = logging.getLogger(__name__)


def get_delivery_info(email):
    r = EmailInfo(email).as_dict()
    
    if not r:
        r = {'status': 'error', 'error': 'no entity with email %s' % email}
        return r, 410, None

    r['status'] = 'ok'
    # Немного магии в Cache-control
    # В рассылке staff, например, 2739 подписчиков, она будет кэшироваться на 500 секунд
    # Маленькие рассылки будут кэшироваться на 60 сек
    h = (r['type'] == 'maillist') and {
        "Cache-Control": "public, max-age=%d" % max(len(r['subscribers']['inbox']) * 0.20, 60)
    }
    return r, 200, h


def load_from_cache(email):
    cached = get_cache_by_email(email)
    if cached and cached.version != FORMAT_VERSION:
        cached = None
    if cached is None:
        return None, None, None

    headers = cached.load_headers() or {}
    headers['X-ML-Cached-On'] = cached.modified_at.isoformat('T')
    body = json.loads(cached.body)
    status_code = 410 if body.get('error') else 200
    return cached.body or '{}', status_code, headers


def save_to_cache(email, body, headers):
    #headers = headers and json.dumps(headers) or None
    o, created = NwsmtpInfoCache.objects.get_or_create(email=email)
    o.version = FORMAT_VERSION
    o.save_headers(headers)
    o.save_body(body)
    o.save()


def _json_response(data, status_code=200, headers=None):
    if not isinstance(data, basestring):
        data = json.dumps(data)
    resp = HttpResponse(data, mimetype='application/json', status=status_code)
    if headers:
        for k, v in headers.items():
            resp[k] = v
    return resp


@require_GET
@prefer_slave
# ML-1559: Отменяем действие глобальных настроек ATOMIC_REQUESTS: True
# для обеих баз в settings.DATABASES, так как db_router сам решает, откуда читать,
# и чтобы не пытаться открывать ненужные транзакции на недоступных хостах.
@transaction.non_atomic_requests('default')
@transaction.non_atomic_requests('slave')
def delivery_info(request):
    """
    Отдаём в nwsmtp информацию, необходимую для доставки письма. JSON.

    Параметры:
    email - адрес, обязательный параметр

    Что возвращаем:

    Если это рассылка:
    * свойства открытости (потому что nswmtp должен баунсить внешние письма в закрытые рассылки)
    * список тех, кто может писать в рассылку (потому что nswmtp должен баунсить письмо, если отправитель не имеет права писать)
    * раскрытый список подписчиков в inbox (потому что nswmtp должен разложить письма в ящики inbox-подписчиков)


    Если это пользователь:
    * ящик, в который класть почту для этого ящика

    Если ни пользователь, ни рассылка не существуют, метод возвращает http-статус 410 (Gone)
    """

    email = request.GET.get('email')
    ignore_cache = request.GET.get('ignore_cache')
    update_cache = request.GET.get('update_cache')

    if email is None:
        return _json_response({'status': 'error', 'error': 'email required'}, status_code=400)

    email = normalize_email(email)

    response, code, headers = None, None, None

    if not ignore_cache and not update_cache:
        response, code, headers = load_from_cache(email)

    if not response:
        response, code, headers = get_delivery_info(email)

    if update_cache and code == 200:
        try:
            with transaction.atomic():
                save_to_cache(email, response, headers)
        except DatabaseError:
            logger.error("Can't update NWSMTP cache, database error")
    return _json_response(response, status_code=code, headers=headers)


def force_update_cache(email):
    email = normalize_email(email)

    # обновить кэш только если код ответа == 200.
    response, code, headers = get_delivery_info(email)
    save_to_cache(email, response, headers)
