# -*- coding: utf-8 -*-
import logging

from django.db import transaction
from django.contrib.auth.models import User
from django.utils.translation import ugettext_lazy as _


import json

from django_intranet_stuff.models import Log, Group

try:
    from django_intranet_stuff.models import ACTION_SAVE, ACTION_DELETE
except ImportError:
    from django_intranet_stuff.models.log import ACTION_SAVE, ACTION_DELETE


from mlcore.ml.models import MailList

from mlcore.permissions.models import GroupPermission
from mlcore.permissions.permission import get_group_list_permissions
from mlcore.permissions.notifications import UnsubscribedBecauseNotMember
from mlcore.permissions.signals import group_membership_changed
from mlcore.tasks.operations import unsubscribe_group, unsubscribe, one_notify_unsubscribed

logger = logging.getLogger(__name__)

GROUP_DELETED_REASON = u"Группа, имеющая доступ, была удалена"


def process_deleted_groups(groups_log):
    groups_pk = groups_log.values_list('primary_key', flat=True).distinct()

    permissions_to_delete = GroupPermission.objects.filter(
        group__id__in=groups_pk, group__intranet_status=0).select_related('list')

    for gp in permissions_to_delete:
        async_res = unsubscribe_group.delay(
            {'initiator': 'auto-usubscribe',
             'comment': u'Удалена группа %s' % gp.group.name},
            gp.group, gp.list, GROUP_DELETED_REASON)

        # Так как celery.chord работает через универсальный интерфейс
        # именуемый в дальнейшем "жопа" приходится изголяться и
        # удалять пермишены синхронно по звершении всех отписок
        async_res.collect()
        gp.delete()


def parse_log_data(center_log_entry_str):
    try:
        center_log_entry = json.loads(center_log_entry_str)
    except ValueError:
        logger.warning("Log context parsing failed: '%s'", center_log_entry_str)
        return

    if not isinstance(center_log_entry, list):
        logger.warning("Log context parsing failed: '%s'", center_log_entry_str)
        return

    center_log_entry = center_log_entry[0]

    if not isinstance(center_log_entry, dict):
        logger.warning("Log context parsing failed: '%s'", center_log_entry_str)
        return

    fields = center_log_entry['fields']
    return fields


def get_affected_memberships(memberships_log):
    ret = {}
    for deleted_membership_data_str in memberships_log.values_list('data', flat=True):
        fields = parse_log_data(deleted_membership_data_str)
        if fields is None:
            continue
        groups = ret.get(fields['staff'])
        if groups is None:
            ret[fields['staff']] = groups = []
        groups.append(fields['group'])

    return ret


def process_deleted_memberships(memberships_log):
    affected_members = get_affected_memberships(memberships_log)

    for user_id, group_ids in affected_members.iteritems():
        user = User.objects.get(staff=user_id)
        # ML-1286: Дополнительно проверяем группы, в логе может быть удаление, но при этом связь есть
        groups = Group.objects.filter(id__in=group_ids).exclude(groupmembership__staff=user_id)
        # Если у пользователя остались другие группы, которые дают доступ в те же рассылки, то не надо отбирать доступ
        lists = MailList.objects.filter(grouppermission__group__in=groups).exclude(
                grouppermission__group__groupmembership__staff=user_id
        )

        # Пытаемся понять в чем проблема ML-1286
        if groups:
            user_groups = Group.objects.filter(groupmembership__staff=user_id)
            for group in user_groups:
                for deleted_group in groups:
                    if group.pk == deleted_group.pk:
                        logger.error("ML-1286: user %s has group %s, but unsubscribed", user_id, group.pk)
        ###############################################################################################################

        str_groups = ', '.join((group.name for group in groups))

        for maillist in lists:
            unsubscribe.delay(
                {
                    'initiator': 'auto-unsubscribe',
                    'comment': u"Отписка бывшего участника группы",
                },
                user,
                maillist
            )


def get_maillists_to_unsubscribe(user):
    group_perms = get_group_list_permissions(user)
    listnames = MailList.objects.filter(is_open=False, subscribers__user=user)\
                            .exclude(owner__user=user)\
                            .exclude(listpermission__user=user,
                                    listpermission__approved=True,
                                    listpermission__type__name='read')\
                            .values_list("name", flat=True)

    return set(listnames).difference(gp.rsplit('.')[0] for gp in group_perms)


def make_job(fun):

    @transaction.atomic
    def job(last_update):
        if last_update is not None:
            time_slice = Log.objects.filter(modified_at__gt=last_update)
        else:
            time_slice = Log.objects.all()

        if not time_slice.exists():
            return last_update

        last_log_update = time_slice.only('modified_at')\
                        .latest('modified_at').modified_at
        
        fun(time_slice)

        return last_log_update

    return job


def _synchronize_center_logs(center_log_slice):
    deleted_groups = center_log_slice\
                     .filter(model_name='Group',
                             data__contains='"intranet_status": 0',
                             action=ACTION_SAVE)
    memberships = center_log_slice\
                          .filter(model_name='GroupMembership')
    deleted_memberships = memberships\
                          .filter(action=ACTION_DELETE)

    logging.info("Unsubscribing users from deleted groups who has no permissions now")
    process_deleted_groups(deleted_groups)
    logging.info("Unsubscribing users who lost permissions due to lost membership in groups")
    process_deleted_memberships(deleted_memberships)


def _import_group_membership_log(center_log_slice):
    '''
    Import membership-related log entries from center to our DB to add fk's
    '''
    for log in center_log_slice\
            .filter(model_name='GroupMembership')\
            .values('action', 'data', 'modified_at'):
        data = parse_log_data(log['data'])
        if data is None:
            continue
        user = data['staff']
        group = data['group']
        # this signal is watched by log-writing app
        group_membership_changed.send(sender=__name__,
                action=log['action'], modified_at=log['modified_at'],
                staff_id=user, group_id=group)


synchronize_center_logs = make_job(_synchronize_center_logs)
import_group_membership_log = make_job(_import_group_membership_log)
