# -*- coding: utf-8 -*-

import settings
from django.core.management.base import BaseCommand
from mlcore.ml.models import LoginRedirect, Subscribers
from mlcore.utils.getters import get_user
from django_intranet_stuff.models import Staff
from mlcore.interaction.compat import compat_requests_session
from django.utils.encoding import force_unicode
from django.conf import settings
from lxml import etree
from mlcore.utils.tvm2 import get_tvm_2_header

import logging

logger = logging.getLogger(__name__)


def get_staff(login):
    try:
        return Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        logger.warning("Login %s not found in staff", login)


class Command(BaseCommand):
    help = 'Load redirects from staff'

    def load_xml(self, url):
        session = compat_requests_session()
        headers = get_tvm_2_header(settings.STAFF_TVM2_ID)
        response = session.get(url, headers=headers)
        try:
            return etree.fromstring(response.content)
        except etree.XMLSyntaxError:
            logger.error(u'Invalid xml from %s:\n%s', url,
                         force_unicode(response.content, errors=u'ignore'))
            raise

    def parse_xml(self, xml):
        for el in xml.getchildren():
            login_from = el.find('from').text
            for login_to in set([login.text for login in el.find('to')]):
                yield login_from, login_to

    def update_maillists(self, r):
        user = get_user(r.staff)
        subscribers = Subscribers.objects.filter(user=user)
        for s in subscribers:
            s.list.set_as_modified()

    def handle(self, *args, **options):

        existing_redirects_login = {}
        for r in LoginRedirect.objects.select_related('staff').all():
            staff_redirects = existing_redirects_login.setdefault(r.staff, {})
            staff_redirects[r.redirect_to] = r

        xml = self.load_xml(url=settings.STAFF_URL_API + '?token=ml')

        for login_from, login_to in self.parse_xml(xml):
            from_ = get_staff(login_from)
            to_ = get_staff(login_to)

            if from_ and to_:
                is_new = True
                redirect_from = existing_redirects_login.get(from_, None)
                if redirect_from:
                    redirect_to = redirect_from.get(to_, None)
                    if redirect_to:
                        existing_redirects_login[from_].pop(to_)
                        is_new = False

                if is_new:
                    redirect = LoginRedirect(staff=from_, redirect_to=to_)
                    redirect.save()
                    self.update_maillists(redirect)

        for redirect_from in existing_redirects_login.itervalues():
            for redirect in redirect_from.itervalues():
                redirect.delete()
                self.update_maillists(redirect)
