# -*- coding: utf-8 -*-

import logging

from django.db import models
from django.contrib.auth.models import User

logger = logging.getLogger(__name__)


class ListPermissionManager(models.Manager):

    def requests_count(self):
        return self.requests().count()

    def requests(self):
        '''
        Get all permission requests waiting for approvement.
        If list became open, requests are skipped.
        '''
        return self.filter(approved=False, list__is_open=False)

    def old_requests(self):
        from datetime import date, timedelta
        return self.requests().filter(modified__lt=date.today() - timedelta(2))

    def grant_permission(self, maillist, user=None, permission_type=None, user_id=None):
        '''
        Create or get a permission object for given list and user and mark it
        as approved.
        '''
        if permission_type is None:
            from mlcore.permissions.models import Type
            permission_type = Type.objects.get(name='read')

        if user is None:
            if user_id is None:
                raise ValueError("user and user_id cannot be both None")
            user = User(id=user_id)

        perm, created = self.get_or_create(
                list=maillist, user=user, type=permission_type,
                defaults={'approved': True})
        if not created and not perm.approved:
            perm.approved = True
            perm.save()
        logger.info(u"Granted (or checked that is was already granted) permission: %s", perm)
        return perm

    def get_by_natural_key(self, username, listname, type_id):
        return self.get(
                user__username=username,
                list__name=listname,
                type__id=type_id)


class GroupPermissionManager(models.Manager):

    def get_by_natural_key(self, group_url, listname, type_id):
        return self.get(
                group__url=group_url,
                list__name=listname,
                type__id=type_id)
