# -*- coding: utf-8 -*-

from django.conf import settings
from mlcore.emails import Notification


class AskPermission(Notification):
    template_type = 'ask_permission.html'

    def __init__(self, operation):
        self.operation = operation
        self.user = operation.user
        self.list = operation.list
        self.context = {
            "user": self.user,
            "list": self.list,
        }

    def send(self):
        recipients = []
        if self.list.owner_set.count() > 0:
            for owner in self.list.owner_set.select_related("user").all():
                if owner.user.email:
                    recipients.append(owner.user.email)
        else:
            recipients = settings.MAILLIST_DEFAULT_OWNERS

        sender = settings.SERVICE_EMAIL
        reply_to = (u'%s %s' % (self.user.first_name,
                                self.user.last_name), self.user.email)
        super(AskPermission, self).send(recipients, sender, reply_to=reply_to)


class AskPermissionWithSubscription(Notification):
    template_type = 'ask_permission_with_subscription.html'

    def __init__(self, operation):
        self.operation = operation
        self.user = operation.user
        self.list = operation.list
        self.type = operation.type
        self.context = {
            "user": self.user,
            "list": self.list,
            "type": self.type,
            "type_code": self.get_type_code(),
        }

    def get_type_code(self):
        if self.type == "inbox":
            return "SUB_INBOX"
        if self.type == "imap":
            return "SUB_IMAP"
        if self.type == "both":
            return "SUB_INBOX + SUB_IMAP"
        return "none"

    def send(self):
        recipients = []
        if self.list.owner_set.count() > 0:
            for owner in self.list.owner_set.select_related("user").all():
                if owner.user.email:
                    recipients.append(owner.user.email)
        else:
            self.context["to_default_owners"] = True
            recipients = settings.MAILLIST_DEFAULT_OWNERS

        sender = settings.SERVICE_EMAIL
        reply_to = (u'%s %s' % (self.user.first_name,
                                self.user.last_name), self.user.email)
        super(AskPermissionWithSubscription, self).send(recipients,
                                                        sender,
                                                        reply_to=reply_to)


class AskPermissionForFriend(AskPermissionWithSubscription):
    template_type = 'ask_permission_for_friend.html'

    def __init__(self, operation):
        super(AskPermissionForFriend, self).__init__(operation)
        self.context.update(operation.context)

class PermissionApproved(Notification):
    template_type = 'permission_approved.html'


class PermissionDenied(Notification):
    template_type = 'permission_denied.html'


class PermissionRevoked(Notification):
    template_type = 'permission_revoked.html'


# notification for users that were unsubscribed because group permission that
# gave them access to list was revoked
class GroupPermissionRevoked(Notification):
    template_type = 'group_permission_revoked.html'


class UnsubscribedBecauseNotMember(Notification):
    template_type = 'unsubscribed_because_not_member.html'


class GrantedAccess(Notification):
    template_type = 'granted_access.html'


class PermissionReminder(Notification):
    """
    Notification to send a list of requests to one owner.
    All the requests should be to maillists with this owner.
    If owner is not provided, notification will be sent to default owners.
    Requests are grouped by lists
    """
    template_type = 'permission_reminder.html'

    def __init__(self, requests, owner=None):
        requests = sorted(requests, key=lambda r: r.list.name)
        self.list2requests = self._get_list2requests(requests)
        self.requests = requests
        self.owner = owner
        self.context = {
            "list2requests": self.list2requests
        }

    def _get_list2requests(self, requests):
        l2r = {}
        for request in requests:
            if not request.list in l2r:
                l2r[request.list] = []
            # если рассылка отмечена, как удаленная, то не напоминаем о запросах на нее (ML-1275)
            if not request.list.is_deleted:
                l2r[request.list].append(request)
        return l2r

    def send(self):
        recipients = []
        if self.owner:
            recipients.append(self.owner.user.email)
        else:
            recipients = settings.MAILLIST_DEFAULT_OWNERS

        sender = settings.SERVICE_EMAIL
        super(PermissionReminder, self).send(recipients, sender)

