# -*- coding: utf-8 -*-
import logging

from django.contrib.auth.models import User
from django_intranet_stuff.models import Group

from mlcore.subscribe import subscription_type

from mlcore.permissions.models import GroupPermission
from mlcore.permissions.signals import group_permission_revoked
from mlcore.permissions.notifications import GroupPermissionRevoked
from mlcore.permissions.utils import group_permitted_usernames,\
         first_permitted_group_for_user
from mlcore.permissions.revoke.user import revoke_and_unsubscribe

logger = logging.getLogger(__name__)


class NoPermissionsToRevokeError(Exception):
    pass


def process_groups_info(list, group_ids):
    context = {}
    group_permissions = context['group_permissions'] = GroupPermission.objects\
                        .select_related()\
                        .filter(group__id__in=group_ids, list=list)

    # hit db once - on len() evaluation
    context['group_permissions_count'] = group_permissions_count = len(group_permissions)
    if not group_permissions_count:
        return context

    # set with owner logins
    owners = set(list.owner_set.values_list('user__username', flat=True))
    # mapping usernames -> subscription types
    subscribers = dict(list.subscribers_set.values_list('user__username', 'stype'))
    # get groups we want to take permissions from
    groups = [gp.group for gp in group_permissions]
    groups_ids = [gp.group_id for gp in group_permissions]
    # with permission
    users_with_permissions = set(list.listpermission_set.filter(approved=True, type__name='read').values_list('user__username', flat=True))
    # from other groups that have permissions
    people_from_other_permitted_groups = set(group_permitted_usernames(list, groups_ids)
                                             .values_list('login', flat=True))

    permission_preserved = []
    to_unsubscribe = []
    to_revoke = []
    affected_users = {}

    for g in groups:

        for (member_login, last_name, first_name) in g.get_all_members()\
                .order_by('last_name', 'first_name')\
                .values_list("login", "last_name", "first_name"):

            cached_user = affected_users.get(member_login)
            if cached_user:
                cached_user['affected_groups'].append(g)
                continue

            full_name = (u"%s %s" % (
                        last_name if last_name is not None else '',
                        first_name if first_name is not None else '')).strip()

            affected_users[member_login] = user_ext = {
                'affected_groups': [g],
                'username': member_login,
                'fullname': full_name,
            }

            user_ext['subscription_type'] = subscribers.get(member_login, subscription_type.NONE)
            # dispatch user to appropriate dict
            if member_login in owners:#self._is_user_owner(user):
                user_ext['is_owner'] = True
                user_ext['reason_to_preserve'] = 'is_owner'
                permission_preserved.append(user_ext)
            elif member_login in people_from_other_permitted_groups:
                user_ext['reason_to_preserve'] = 'permitted_groups'
                # save info about permitted groups which permissions would not be revoked
                group_id = first_permitted_group_for_user(member_login, list, groups_ids)
                group_id = group_id.id if group_id else None
                user_ext['permitted_groups'] = [Group.objects.only('name').get(id=group_id)]
                permission_preserved.append(user_ext)
            elif member_login in users_with_permissions:#permissions.filter(list=list, approved=True).exists():
                user_ext['reason_to_preserve'] = 'has_personal_permission'
                permission_preserved.append(user_ext)
            elif user_ext['subscription_type'] != subscription_type.NONE:
                # user has personal subscription
                to_unsubscribe.append(user_ext)
            else:
                to_revoke.append(user_ext)

    context['to_revoke'] = to_revoke
    context['to_unsubscribe'] = to_unsubscribe
    context['permission_preserved'] = permission_preserved
    context['affected_users'] = affected_users
    return context


def revoke_group_permissions(list, group_ids,
        notification_fabric=GroupPermissionRevoked, force_pending=True,
        by_owner=None, group_deleted=False,
        **extra_context):
    context = process_groups_info(list, group_ids)

    if not context['group_permissions_count']:
        return

    # TODO: invalidate group permission cache for affected users
    #affected_users_logins = [u.username for u in context['affected_users']]

    # unsubscribe users
    to_unsubscribe = context['to_unsubscribe']
    to_unsubscribe_users = User.objects.filter(username__in=(ue['username'] for ue in to_unsubscribe))

    logger.info(u"Unsubscribing people because we revoked access from "
            "groups (group ids %s)", group_ids)
    revoke_and_unsubscribe(list,
            to_unsubscribe_users,
            notification_fabric=notification_fabric,
            force_pending=force_pending,
            by_owner=by_owner,
            group_deleted=group_deleted,
            **extra_context)

    GroupPermission.objects.filter(group__id__in=group_ids).delete()
    for gp in context['group_permissions']:
        group_permission_revoked.send_robust(sender=None,
                group_id=gp.group_id, list_id=gp.list_id,
                by_owner=by_owner,
                group_deleted=group_deleted)
