# -*- coding: utf-8 -*-

import logging

from django.conf import settings
from django.utils.translation import ugettext

from mlcore.subscribe.user import unsubscribe_user

from mlcore.permissions.signals import user_access_revoked, permission_denied
from mlcore.permissions.models import ListPermission
from mlcore.permissions.notifications import PermissionRevoked
from mlcore.permissions.utils import first_permitted_group_for_user

logger = logging.getLogger(__name__)


def deny_user_permissions(perms, messager=None):
    """
    Deny user permissions
    """
    for perm in perms:
        perm.delete()
        if messager:
            messager.success(
                ugettext(u'MSG.REVOKE_ACCESS %(username)s %(list_name)s') % {
                    'username' : perm.user.username,
                    'list_name' : perm.list.name
                }
            )
    permission_denied.send(sender=perms)
    return True


def revoke_permission(list, user):
    '''
    Delete user's READ permission for given list if it is exists.

    @rtype: ListPermission object or None (if no permission deleted)
    '''
    logger.info(u"Revoking permission to list %s from user %s", list.name, user.username)
    try:
        permission = user.permissions.get(list=list, type__name="read")
    except ListPermission.DoesNotExist:
        pass
    else:
        permission.delete()
        return permission


def revoke_and_unsubscribe_user(list, user,
        notification_fabric=PermissionRevoked, force_pending=True,
        omit_master=False,
        **extra_context):
    permission = revoke_permission(list, user)
    result = unsubscribe_user(list, user,
            force_pending=force_pending, omit_master=omit_master)
    if result or (permission and not list.is_open):
        extra_context.update({'list': list})
        notification = notification_fabric(extra_context)
        notification.send([user.email], sender=settings.SERVICE_EMAIL)
        user_access_revoked.send(sender=__name__, list=list,
                user=user, extra_context=extra_context)

        logger.info(u"Revoked permission/unsubscribed user %s from list "
                "%s: unsubscribe success=%s, permission was in db: %s",
                user.username, list.name, result, permission)
        return True
    else:
        return False


def revoke_and_unsubscribe_users(list, user_iterable,
        notification_fabric=PermissionRevoked, force_pending=True,
        omit_master=False, **extra_context):
    '''
    For each user in user_queryset we try to delete ListPermission object for
    given list and to unsubscribe user from this list. If we successfully
    unsubscribed or if we deleted permission and list was not open, we sends
    him a notification.

    If force_pending is True, we will save failed backend attemts anyway
    (normally we breakes if cmail returned False)
    '''

    for user in user_iterable:
        revoke_and_unsubscribe_user(list, user,
                notification_fabric=notification_fabric,
                force_pending=force_pending, omit_master=omit_master,
                **extra_context)


def revoke_and_unsubscribe(list, user_queryset, force=False,
        notification_fabric=PermissionRevoked, force_pending=True,
        omit_master=False, **extra_context):
    '''
    High level method to unsubscribe selected users from maillist.
    It will filter out owners and will not unsubscribe people that have
    permissions through groups.

    @param user_queryset: users to unsubscribe (and to take permissions from)
    @param list: MailList object
    @param force: boolean value, if True we omit filtering
    @param **extra_context: additional context for notification template
        rendering and another optional args:
        'reason':  reason of revoking;
        'by_owner': Staff model instance that revoked permission;
        'notification_fabric': one can redefine notification template (must
                have method 'send' and takes context dict at construction);
        'force_operation': force operation to stay pending on fail.
    @returns: generator of tuples (user, reason) with reasons why users stays
        subscribed
    '''

    if force:
        users_to_unsubscribe = user_queryset
    else:
        users_to_unsubscribe = []
        # we are excluding list owners because it is kinda nonsense to deny
        # access from them
        for user in user_queryset.exclude(owner__list=list):
            # check if user is in group that have permission
            group_id = first_permitted_group_for_user(user, list)
            group_id = group_id.id if group_id else None
            if group_id is None:
                users_to_unsubscribe.append(user)
            else:
                # user has access through group, so we are just trying to delete his
                # personal permission if there is one
                revoke_permission(list, user)
                # and issue a warning message about him
                yield (user, group_id)

    # finally, unsubscribing users that have no group access
    revoke_and_unsubscribe_users(list, users_to_unsubscribe,
            notification_fabric=notification_fabric, force_pending=force_pending,
            omit_master=omit_master,
            **extra_context)

