# coding: utf-8

from operator import or_

from django.db.models import Q
from django.contrib.auth.models import User
from django.core.exceptions import ObjectDoesNotExist
from django_intranet_stuff.models import Group, Staff
from mlcore.permissions.models import Type


def get_groups_by_q(q_gen):
    q = reduce(or_, q_gen, Q())

    groups_qs = (Group.objects
                 .filter(q)
                 .filter(level__gt=0, intranet_status=1))

    return groups_qs


def get_parent_groups(groups_qs):
    """ Берет queryset групп и возвращает
    queryset этих групп с надгруппами, никакой порядок не гарантируется
    """
    my_groups_qs = groups_qs.filter(intranet_status=1).values('lft', 'rght', 'tree_id')
    if not my_groups_qs:
        return my_groups_qs.none()
    q_gen = (Q(lft__lte=g['lft'], rght__gte=g['rght'],
               tree_id=g['tree_id']) for g in my_groups_qs)

    return get_groups_by_q(q_gen)


def get_subgroups(groups_qs):
    """ Берет queryset групп и возвращает
    queryset этих групп с подгруппами, никакой порядок не гарантируется
    """
    my_groups_qs = groups_qs.filter(intranet_status=1).values('lft', 'rght', 'tree_id')
    if not my_groups_qs:
        return my_groups_qs.none()
    q_gen = (Q(lft__gte=g['lft'], rght__lte=g['rght'],
               tree_id=g['tree_id']) for g in my_groups_qs)

    return get_groups_by_q(q_gen)


def groups_permitted_for_list(maillist):
    '''
    Return iterable of group id's that are permitted to read list lst due to
    group permissions.
    '''
    groups_id = maillist.grouppermission_set.values_list('group', flat=True)
    groups = Group.objects.filter(id__in=groups_id)
    return get_subgroups(groups)


def group_permitted_usernames(maillist, group_ids_to_exclude=()):
    '''
    Return iterable of usernames that are permitted to read list lst due to
    group permissions.
    '''
    groups = groups_permitted_for_list(maillist).exclude(
        id__in=group_ids_to_exclude)
    return Staff.objects.filter(id__in=groups.values('members'))


def first_permitted_group_for_user(user, maillist, group_ids_to_exclude=()):
    '''
    Return first group that allows user to read maillist lst or None if there
    are no such group.
    '''
    if isinstance(user, basestring):
        user = Staff.objects.get(login=user)
    elif isinstance(user, User):
        user = user.staff
    elif isinstance(user, Staff):
        pass
    else:
        raise ValueError("Unknown type of user %s" % type(user))

    qs = (Group.objects.filter(permissions__list=maillist,
                               pk__in=get_subgroups(user.in_groups.all()))
          .exclude(id__in=group_ids_to_exclude))
    try:
        return qs[0]
    except IndexError:
        return None


def get_permitted_staff(maillist, permission_type=None):
    groups = maillist.grouppermission_set.all()
    s2 = Staff.objects.filter(user__owner__list=maillist)

    if permission_type:
        type = Type.objects.get(name=permission_type).id
        s1 = Staff.objects.filter(user__permissions__list=maillist, user__permissions__type_id=type)
        group_ids = list(groups.filter(type_id=type)
                         .values_list('group', flat=True))
    else:
        s1 = Staff.objects.filter(user__permissions__list=maillist)
        group_ids = list(groups.values_list('group', flat=True))

    group_ids = list(get_subgroups(
        Group.objects.filter(id__in=group_ids)).values_list('id', flat=True))

    s3 = Staff.objects.filter(groupmembership__group_id__in=group_ids)

    res = set()
    for s in (s1, s2, s3):
        res.update(s.values_list('id', flat=True))

    return Staff.objects.filter(id__in=res)


def get_permitted_folders(user, permission_type=None):
    from mlcore.ml.models import MailList
    from mlcore.utils.getters import get_staff

    staff = get_staff(user)

    q1 = dict(listpermission__user_id=staff.user_id,
              listpermission__approved=True)
    q2 = dict(owner__user_id=staff.user_id)
    grp = set(get_parent_groups(staff.in_groups.all()).values_list('id', flat=True))
    q3 = dict(grouppermission__group_id__in=grp)

    if permission_type:
        type = Type.objects.get(name=permission_type).id
        q1['listpermission__type_id'] = type
        q3['grouppermission__type_id'] = type

    ids = set()
    for q in (q1, q2, q3):
        ids.update(MailList.objects.filter(**q).values_list('id', flat=True))

    return MailList.objects.filter(id__in=ids)


def get_group_permissons(user):
    from mlcore.ml.models import MailList
    if isinstance(user, basestring):
        user = Staff.objects.get(login=user)
    elif isinstance(user, User):
        user = user.staff
    elif isinstance(user, Staff):
        pass
    else:
        raise ValueError("Unknown type of user %s" % type(user))

    return MailList.objects.filter(
        grouppermission__group__in=get_subgroups(user.in_groups.all())).distinct()


def is_external_staff(user):
    from mlcore.utils.getters import get_staff
    try:
        return get_staff(user).affiliation == 'external'
    except ObjectDoesNotExist:
        # В рассылке qa был юзер без записи в стафе, этот метод валил вьюху
        return False


def can_read(user, maillist):
    """Проверяем может ли пользователь читать рассылку. Если он на нее
    подписан или владелец, или рассылка не ридонли, или у него или его
    группы явно есть права значит может.
    Если сотрудник "внешний", то проверяем наличие галки external_staff_can_read.
    """

    from mlcore.ml.models import Subscribers
    from mlcore.utils.getters import get_user, get_list

    user = get_user(user)
    maillist = get_list(maillist)

    # SWATSUP-586 Кешируем работу этой функции, потому что в большинстве случаев она делает одно и тоже
    if not hasattr(user, 'maillist_can_read_cache'):
        setattr(user, 'maillist_can_read_cache', dict())

    if maillist.id in user.maillist_can_read_cache:
        return user.maillist_can_read_cache[maillist.id]
    # ML-1119: вводим новый уровень прав
    # Запрещаем читать рассылку внешним сотрудникам, если не стоит галочка external_staff_can_read
    elif (not maillist.external_staff_can_read) and is_external_staff(user):
        result = False
    elif maillist.is_open:
        result = True
    elif Subscribers.objects.filter(user=user, list=maillist).exists():
        result = True
    else:
        result = get_permitted_folders(user, 'read').filter(id=maillist.id).exists()

    user.maillist_can_read_cache[maillist.id] = result
    return result


def can_write(user, maillist):
    """Проверяем может ли пользователь писать в рассылку. Может если
    рассылка не readonly, если он владелец или если у него или группы
    есть доступ.

    """
    from mlcore.utils.getters import get_user, get_list

    maillist = get_list(maillist)

    if not maillist.readonly:
        return True

    user = get_user(user)

    return get_permitted_folders(user, 'write').filter(id=maillist.id).exists()
