# encoding: utf-8
from __future__ import unicode_literals
import logging

from blackbox import BlackboxError
from django.conf import settings

from mlcore.utils.blackbox_util import get_uid, get_userinfo
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext

from .exceptions import PGMigrationError


def _get_maillist_replica_login(maillist):
    try:
        context = YandexTeamBackendContext.objects.get(maillist=maillist)
        return get_replica_login(context.passport_name)
    except YandexTeamBackendContext.DoesNotExist:
        logging.error('YandexTeam context not found for %s', maillist)
        raise PGMigrationError('Can\'t get y-t context for %s to resolve replica login', maillist.name)


def get_replica_login(imap_name):
    return '{}{}'.format(imap_name, settings.PG_MIGRATION_SUFFIX)


def get_replica_email(imap_name):
    return '{}@yandex-team.ru'.format(get_replica_login(imap_name))


def get_maillist_replica_email(maillist):
    return '{}@yandex-team.ru'.format(_get_maillist_replica_login(maillist))


def get_maillist_replica_uid(maillist, raise_on_error=True):
    replica_login = _get_maillist_replica_login(maillist)
    uid = get_uid(replica_login)
    if not uid and raise_on_error:
        raise PGMigrationError('Maillist PG replica not found in blackbox for %s', maillist.name)

    return uid


def user_is_pg(user):
    uid = user.staff.uid
    user_info = get_userinfo(uid, by_login=False)

    return user_info.get('fields', {}).get('mdb') == 'pg'


def maillist_is_pg(maillist):
    try:
        context = YandexTeamBackendContext.objects.get(maillist=maillist)
        if context.uid:
            user_info = get_userinfo(context.uid, by_login=False)

            return user_info['fields'].get('mdb') == 'pg'

    except YandexTeamBackendContext.DoesNotExist:
        logging.error('YandexTeam context not found for %s', maillist)
        raise PGMigrationError('Can\'t get y-t context for %s to resolve login', maillist.name)
    except BlackboxError:
        logging.error('BlackBox error for %s', maillist)
        raise PGMigrationError('Can\'t get mdb for %s to resolve login', maillist.name)

    return False

