import logging
import datetime

from django_intranet_stuff.models import Staff, Department
from django.conf import settings
from django.db.models import Q

from mlcore.subscribe.operations.operations import EmailSubscribe, EmailUnsubscribe
from mlcore.ml.models import MailList

from models import *

class SMSFilterError(Exception):
    '''Base class for all filter errors'''

class EmptyFilter(SMSFilterError):
    '''No filtering conditions'''

class InvalidFilter(SMSFilterError):
    '''Invalid Filter'''

class WrongMailList(SMSFilterError):
    '''MailList doesn't support SMS'''


def _filter_queryset(queryset, filter):

    if filter.type == 'car':
        condition = not (filter.condition.strip() in ['1', 'True', 'true', 'TRUE'])
        return queryset.filter(car__isnull=condition)

    if filter.type == 'department':
        try:
            departments = [int(s) for s in filter.condition.split(',')]
        except ValueError:
            raise InvalidFilter('Invalid department id')
        
        all_departments = []

        for root in departments:
            all_departments.append(root)
            try:
                department = Department.objects.get(id=root)
            except Department.DoesNotExist:
                raise InvalidFilter('Department %s isn\'t found')
            all_departments.extend([d.id for d in department.get_descendants()])
       
        all_departments = list(set(all_departments))

        return queryset.filter(department__in=all_departments)

    if filter.type == 'office':
        try:
            offices_id = [int(s) for s in filter.condition.split(',')]
        except ValueError:
            raise InvalidFilter('Invalid office id')
        return queryset.filter(office__in=offices_id)
    
    if filter.type == 'position':
        patterns = [s.strip() for s in filter.condition.split(',')]
        q = Q()
        for pattern in patterns:
            if pattern:
                if pattern[0] == '!':
                    q |= Q(position__iexact=pattern[1:])
                else:
                    q |= Q(position__icontains=pattern)
        return queryset.filter(q)

def get_sms_mail_list_logins(sms_mail_list):
    users = Staff.objects.filter(is_dismissed=False).exclude(department__id__in=settings.SMS_EXCLUDE_DEPARTMENTS)

    filters = list(sms_mail_list.filters.all())
    if not filters:
        raise EmptyFilter('Filter list is empty')
    
    for filter in filters:
        users = _filter_queryset(users, filter)
    
    return [user.login for user in users]

def update_mail_list(mail_list):
    try:
        if not mail_list.is_sms:
            raise WrongMailList("Mail List doesn't support sms")

        emails_list = []
        for sms_mail_list in mail_list.smsmaillist_set.all():
            try:
                emails_list.extend([login + settings.SMS_MAIL_PREFIX \
                        for login in get_sms_mail_list_logins(sms_mail_list)])
            except EmptyFilter:
                pass
        emails = set(emails_list)

        existing_emails = set([es.email for es in \
                mail_list.emailsubscriber_set.all() \
                if es.email.endswith(settings.SMS_MAIL_PREFIX)])
        
        new_emails = emails - existing_emails
        obsolete_emails = existing_emails - emails

        for email in new_emails:
            op = EmailSubscribe(sms_mail_list.mail_list, email)
            op.perform()

        for email in obsolete_emails:
            op = EmailUnsubscribe(sms_mail_list.mail_list, email)
            op.perform()
    
        mail_list.smsmaillist_set.all().update(last_update=datetime.datetime.now())
        logging.info('Updated maillist %s (%s)' % (mail_list.name, mail_list.id))
        
        return True, 'ok'

    except SMSFilterError, error:
        logging.error('Error while updating sms maillist %s (%s): %s' % (mail_list.name, mail_list.id, error.message))
        return False, error.message

def update_sms_mail_lists():
    for mail_list in MailList.objects.filter(is_sms=True):
        update_mail_list(mail_list)
