# -*- coding: utf-8 -*-
import logging
from django.db import models
from django.contrib.auth.models import User
from django.db.utils import IntegrityError
from django.utils.translation import ugettext_lazy as _

from mlcore.subscribe import subscription_type
from mlcore.subscribe.operations.list import SubscriptionOperationListType,\
         SubscriptionOperationList
from mlcore.ml.models import MailList
from managers import SubscribeActivityManager

class DbLogBase(models.Model):

    class Meta:
        abstract = True
        get_latest_by = 'datetime'

    datetime = models.DateTimeField(auto_now_add=True)
    log_type = None


class SubscribeOperationLogManager(models.Manager):

    def get_last_joined(self, list, limit, offset=0):
        '''
        Return limit-offset subscribe operation log entries with successive
        subscribe operations for specified list.
        '''
        return (self.filter(list=list,
                    operation_type=SubscriptionOperationListType.SUBSCRIBE,
                    list__subscribers__user=models.F('subscriber'),
                    list__subscribers__user__is_active=True,
                    operation_result_code=1)
                        .order_by('datetime')
                        .select_related(
                            'subscriber__id', 'subscriber__last_name',
                            'subscriber__first_name', 'subscriber__username',
                            'meddler__id', 'meddler__last_name',
                            'meddler__first_name',
                            'meddler__username')[offset:limit+offset])


class SubscribeOperationLog(DbLogBase):

    log_type = 'subscribe_operation'

    TYPE_CHOICES = subscription_type.composite_types.items()

    subscriber = models.ForeignKey(User)
    meddler = models.ForeignKey(User, null=True,
        related_name='subscribeoperationlog_via_meddler_set')

    list = models.ForeignKey(MailList)
    old_type = models.CharField(max_length=32,
            choices=TYPE_CHOICES)
    new_type = models.CharField(max_length=32,
            choices=TYPE_CHOICES)
    operation_type = models.CharField(max_length=32,
            choices=SubscriptionOperationListType.choices)
    operation_result_code = models.IntegerField(default=0)

    objects = SubscribeOperationLogManager()

    def __unicode__(self):
        return u'[%s on %s] %s -> %s result=%s' % (self.subscriber.username,
                self.list.name, self.old_type, self.new_type,
                self.operation_result_code)

#    @property
#    def user(self):
#        return self.subscriber


class GroupPermissionLog(DbLogBase):

    class Types:

        GRANTED_BY_OWNER = 'granted_by_owner'
        REVOKED_BY_OWNER = 'revoked_by_owner'
        REVOKED_BECAUSE_DELETED = 'revoked_deleted'

        TYPES = (
                GRANTED_BY_OWNER,
                REVOKED_BY_OWNER,
                REVOKED_BECAUSE_DELETED,
                )

        CHOICES = zip(TYPES, TYPES)

    group = models.ForeignKey('django_intranet_stuff.Group')
    list = models.ForeignKey(MailList)
    by_owner = models.ForeignKey(User, blank=True, null=True)
    type = models.CharField(max_length=32, choices=Types.CHOICES)


class AccessManagementLog(DbLogBase):

    log_type = 'access_management'

    class Types:

        GRANTED_BY_OWNER = 'granted_by_owner'
        APPROVED_BY_OWNER = 'approved_by_owner'
        GRANTED_THROUGH_GROUP = 'granted_through_group'
        RESTRICTED_BY_OWNER = 'restricted_by_owner'
        REVOKED_BY_OWNER = 'revoked_by_owner'
        REVOKED_AUTO = 'revoked_auto'

        TYPES = (
                GRANTED_BY_OWNER,
                APPROVED_BY_OWNER,
                GRANTED_THROUGH_GROUP,
                RESTRICTED_BY_OWNER,
                REVOKED_BY_OWNER,
                REVOKED_AUTO,
                )

        CHOICES = (
            (GRANTED_BY_OWNER, _('MSG.ACCESS_LOG_GRANTED_BY_OWNER')),
            (APPROVED_BY_OWNER, _('MSG.ACCESS_LOG_APPROVED_BY_OWNER')),
            (GRANTED_THROUGH_GROUP, _('MSG.ACCESS_LOG_GRANTED_THROUGH_GROUP')),
            (RESTRICTED_BY_OWNER, _('MSG.ACCESS_LOG_RESTRICTED_BY_OWNER')),
            (REVOKED_BY_OWNER, _('MSG.ACCESS_LOG_REVOKED_BY_OWNER')),
            (REVOKED_AUTO, _('MSG.ACCESS_LOG_REVOKED_AUTO')),
            )

    subject = models.ForeignKey(User, related_name='access_log')
    list = models.ForeignKey(MailList)
    manager = models.ForeignKey(User, null=True, blank=True,
            related_name='grants_log')
    type = models.CharField(max_length=32,
            choices=Types.CHOICES)
    info = models.CharField(max_length=512, blank=True)

    @property
    def user(self):
        return self.subject

    def __unicode__(self):
        return u'Access operation: %s; user %d list %d' % (self.type, self.subject_id, self.list_id)


class SubscribeActivity(DbLogBase):
    SUB_TYPES = (
        (1, 'IMAP'),
        (2, 'Inbox'),
    )
    SUB_ACTIONS = (
        (0, 'Unsubscribe'),
        (1, 'Subscribe'),
    )
    WHERES = (
        (1, 'Client'),
        (2, 'Service'),
    )

    list = models.ForeignKey(MailList)
    user = models.ForeignKey(User)
    sub_type = models.PositiveSmallIntegerField(choices=SUB_TYPES)
    sub_action = models.PositiveSmallIntegerField(choices=SUB_ACTIONS)
    where = models.PositiveSmallIntegerField(choices=WHERES)

    objects = SubscribeActivityManager()

    class Meta:
        db_table = 'wakka_subscribeactivity'


class EmailActivity(DbLogBase):
    list = models.ForeignKey(MailList)
    email = models.CharField(max_length=255)
    operation = models.CharField(max_length=20, db_index=True)


class GroupMembershipLog(models.Model):

    class Meta:
        get_latest_by = 'datetime'

    MEMBERSHIP_LOG_TYPE_CHOICES = (
            (0, 'Deleted'),
            (1, 'Added'),
        )

    group = models.ForeignKey('django_intranet_stuff.Group')
    user = models.ForeignKey('django_intranet_stuff.Staff')
    type = models.PositiveIntegerField(choices=MEMBERSHIP_LOG_TYPE_CHOICES)
    datetime = models.DateTimeField()


def connect_signals():

    from mlcore.subscribe.signals import operation_list_performed
    from mlcore.permissions.signals import\
            group_membership_changed,\
            permission_denied, permission_approved,\
            permissions_granted, user_access_revoked,\
            group_permission_revoked, group_permission_granted

    def log_operation_list(sender, **kwargs):
        # filter out not subscribe operations (notifications and such)
        if not issubclass(sender, SubscriptionOperationList):
            return

        instance = kwargs['instance']
        # do not log empty operations
        if instance.isEmpty():
            return

        result = int(kwargs['result'])
        user = instance.getUser()
        # do not log email operations as user operations
        if user is None:
            return

        list = instance.list
        optype = instance.getType()

        meddler = instance.getContext().get('meddler')

        SubscribeOperationLog.objects.create(
                list=list,
                subscriber=user,
                meddler=meddler,
                operation_result_code=result,
                operation_type=optype,
                old_type=instance.from_type,
                new_type=instance.to_type,
        )

    def log_access_approved(sender, **kwargs):
        AccessManagementLog.objects.create(
            type=AccessManagementLog.Types.APPROVED_BY_OWNER,
            manager=None,
            list=sender[0].list,
            subject=sender[0].user,
            )

    def log_access_denied(sender, **kwargs):
        AccessManagementLog.objects.create(
            type=AccessManagementLog.Types.RESTRICTED_BY_OWNER,
            manager=None,
            list=sender[0].list,
            subject=sender[0].user,
            )

    def log_access_granted(sender, **kwargs):
        by_owner = kwargs['by_owner']
        list = kwargs['list']
        users = kwargs['granted_users']
        for user in users:
            AccessManagementLog.objects.create(
                type=AccessManagementLog.Types.GRANTED_BY_OWNER,
                manager=by_owner.django_user,
                list=list,
                subject=user
            )

    def log_access_revoked(sender, **kwargs):
        list_ = kwargs['list']
        user = kwargs['user']
        extra_context = kwargs['extra_context']
        by = extra_context.get('by_owner')
        reason = extra_context.get('reason', '')
        if by is None: # revoked automatically
            type_ = AccessManagementLog.Types.REVOKED_AUTO
        else:
            type_ = AccessManagementLog.Types.REVOKED_BY_OWNER

        AccessManagementLog.objects.create(
            type=type_,
            manager=by.django_user if by is not None else None,
            list=list_,
            subject=user,
            info=reason,
        )

    def log_membership_changed(sender, **kwargs):
        staff_id = kwargs['staff_id']
        group_id = kwargs['group_id']
        modified_at = kwargs['modified_at']
        action = kwargs['action']

        if action == 'd':
            type_ = 0
        elif action == 's':
            type_ = 1
        else:
            return

        type_ = 0 if kwargs['action'] == 'd' else 1
        try:
            GroupMembershipLog.objects.create(
                    user_id=staff_id, group_id=group_id, type=type_,
                    datetime=modified_at)
        except IntegrityError as exc:
            # ML-1315 quick fix: just ignore and log IntegrityError
            logging.warning('Error %s in log_membership_changed. kwargs=%s', exc, kwargs)

    def log_group_permission_revoked(sender, **kwargs):
        log_entry = GroupPermissionLog(
            list_id=kwargs['list_id'],
            group_id=kwargs['group_id'],
            )

        if kwargs.get('group_deleted'):
            log_entry.type = GroupPermissionLog.Types.REVOKED_BECAUSE_DELETED
        else:
            log_entry.type = GroupPermissionLog.Types.REVOKED_BY_OWNER
            if kwargs.get('by_owner') is not None:
                log_entry.by_owner = kwargs['by_owner'].django_user

        log_entry.save()

    def log_group_permission_granted(sender, **kwargs):
        GroupPermissionLog.objects.create(
            type=GroupPermissionLog.Types.GRANTED_BY_OWNER,
            by_owner=kwargs['by_owner'].django_user,
            group_id=kwargs['group_id'],
            list=kwargs['list'],
            )

    group_membership_changed.connect(log_membership_changed, weak=False)
    group_permission_granted.connect(log_group_permission_granted, weak=False)
    group_permission_revoked.connect(log_group_permission_revoked, weak=False)
    operation_list_performed.connect(log_operation_list, weak=False)
    permission_approved.connect(log_access_approved, weak=False)
    permission_denied.connect(log_access_denied, weak=False)
    permissions_granted.connect(log_access_granted, weak=False)
    user_access_revoked.connect(log_access_revoked, weak=False)


connect_signals()
