# -*- coding: utf-8 -*-

from datetime import date, timedelta
from time import strptime

from django.contrib.auth.decorators import login_required
from django.http import Http404, HttpResponse

from mlcore.stats.models import SubscribeActivity
from mlcore.stats.csv_utils import UnicodeWriter

OUTPUT_CSV  = 'csv'

DEFAULT_OUTPUT = OUTPUT_CSV

PERIOD_TODAY      = 'today'
PERIOD_LAST_WEEK  = 'lastweek'
PERIOD_THIS_WEEK  = 'thisweek'
PERIOD_LAST_MONTH = 'lastmonth'
PERIOD_THIS_MONTH = 'thismonth'
PERIOD_SET        = 'set'

PERIODS = (PERIOD_TODAY, PERIOD_LAST_WEEK, PERIOD_THIS_WEEK, PERIOD_LAST_MONTH,
    PERIOD_THIS_MONTH, PERIOD_SET)

DEFAULT_PERIOD = PERIOD_LAST_WEEK


@login_required
def stats(request):
    """
    Выводит статистику по дням по количеству подписок, отписок через сервис
    и почтовые клиенты.
    """
    output = request.GET.get('output', DEFAULT_OUTPUT).lower()
    output = output in ('html', 'csv') and output or DEFAULT_OUTPUT

    headers = False
    if request.GET.get('headers', False):
        headers = True

    begin, end = _get_interval(request.GET)
    res = SubscribeActivity.objects.stats(begin, end)

    if output == OUTPUT_CSV:
        response = HttpResponse(mimetype='text/csv')
        #response['Content-Disposition'] = 'attachment; filename=somefilename.csv'

        writer = UnicodeWriter(response)

        if headers:
            writer.writerow([u'Дата', u'Подписались через сервис', u'Отписались через сервис',
                u'Подписались через клиент', u'Отписались через клиент'])

        for row in res:
            writer.writerow([row['date'], row['service_subscribes'], row['service_unsubscribes'],
                row['client_subscribes'], row['client_unsubscribes']])

        return response
    raise Http404


@login_required
def actions(request):
    """
    Выводит лог действий пользователей.
    """
    output = request.GET.get('output', DEFAULT_OUTPUT).lower()
    output = output in ('html', 'csv') and output or DEFAULT_OUTPUT

    headers = False
    if request.GET.get('headers', False):
        headers = True

    user = request.GET.get('user', None)
    list = request.GET.get('list', None)

    begin, end = _get_interval(request.GET)
    qs = SubscribeActivity.objects.filter(datetime__gte=begin, datetime__lte=end)
    if user:
        qs = qs.filter(staff__login=user)
    if list:
        qs = qs.filter(list__name=list)

    if output == OUTPUT_CSV:
        response = HttpResponse(mimetype='text/csv')
        #response['Content-Disposition'] = 'attachment; filename=somefilename.csv'

        writer = UnicodeWriter(response)

        if headers:
            writer.writerow([u'Дата и время', u'Пользователь', u'Рассылка', u'Действие', u'Способ', u'Где'])

        for row in qs:
            sub_action = _get_choice_value(row.sub_action, SubscribeActivity.SUB_ACTIONS)
            sub_type = _get_choice_value(row.sub_type, SubscribeActivity.SUB_TYPES)
            where = _get_choice_value(row.where, SubscribeActivity.WHERES)
            writer.writerow([row.datetime, row.user.username, row.list.name, sub_action, sub_type, where])
        return response
    raise Http404


def _get_interval(params):
    """
    Возвращает интервал по словарю параметров.

    @param params: Словать параметров
    @param type: dict

    @return: Интервал (begin, end)
    @rtype: tuple
    """
    begin = None
    end = None

    period = params.get('period', DEFAULT_PERIOD)
    if period not in PERIODS:
        period = DEFAULT_PERIOD

    # временной интервал
    if period == PERIOD_SET:
        begin = date(*strptime(params.get('begin', ''), '%Y-%m-%d')[0:3])
        end = date(*strptime(params.get('end', ''), '%Y-%m-%d')[0:3])
    else:
        end = date.today()
        # сегодня
        if period == PERIOD_TODAY:
            begin = end
        # последние 7 дней
        elif period == PERIOD_LAST_WEEK:
            begin = end - timedelta(days=6)
        # текущая неделя, с понедельника
        elif period == PERIOD_THIS_WEEK:
            begin = end - timedelta(days=end.weekday())
        # последние 30 дней
        elif period == PERIOD_LAST_MONTH:
            begin = end - timedelta(days=29)
        # текущий месяц, с 1го числа
        elif period == PERIOD_THIS_MONTH:
            begin = end.replace(day=1)

    return (begin, end)


def _get_choice_value(value, choices):
    """
    Возвращает значение для отображения по коду значения.

    @param value: Код значения
    @type value: mixed

    @param choices: Набор значений
    @type choices: iterable

    @return: Значение для отображения
    @rtype: mixed
    """
    for val, ch in choices:
        if val != value:
            continue
        return ch
    return None
