# -*- coding: utf-8 -*-
import logging

from mlcore.ymail import procedures
from mlcore.ml.models import MailList
from mlcore.subscribe.backends.base import BaseMaillistBackendInterface
from mlcore.subscribe.backends.yandex_team import passport
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext
from mlcore.subscribe.backends.yandex_team.exceptions import EmptyFidError
from mlcore.utils import blackbox_util

logger = logging.getLogger(__name__)


class MaillistBackendInterface(BaseMaillistBackendInterface):

    type = 'yandex_team'
    model_class = YandexTeamBackendContext

    def __init__(self):
        super(MaillistBackendInterface, self).__init__()

    def subscribe_imap(self, backend_context, user):
        logger.info('Subscribing user %s to folder %s',
                user.username, backend_context.fid)
        if not backend_context.fid:
            raise EmptyFidError(backend_context.backend_type, backend_context.id)
        return procedures.subscribe(user.username, backend_context.fid)

    def unsubscribe_imap(self, backend_context, user):
        logger.info('Unsubscribing user %s to folder %s',
                user.username, backend_context.fid)
        if not backend_context.fid:
            raise EmptyFidError(backend_context.backend_type, backend_context.id)
        return procedures.unsubscribe(user.username, backend_context.fid)

    def subscribe_inbox(self, backend_context, user):
        logger.warning("STUB: yandex_team backend doesn't support inbox subscriptions")
        return True

    def unsubscribe_inbox(self, backend_context, user):
        logger.warning("STUB: yandex_team backend doesn't support inbox subscriptions")
        return True

    def create_maillist(self, backend_context):
        maillist_name = backend_context.passport_name
        maillist = backend_context.maillist
        default_email = maillist.email
        passport.create_passport_maillist(maillist_name, default_email)
        logger.info('Creating maillist in oracle: %s with default_email %s', maillist_name, default_email)
        procedures.create_list(
                maillist_name,
                info=maillist.info,
                is_open=maillist.is_open)

        fsuid = blackbox_util.get_suid(backend_context.passport_name)
        uid = blackbox_util.get_uid(backend_context.passport_name)
        if fsuid:
            MailList.objects\
                .filter(pk=maillist.pk).update(fsuid=fsuid)
            YandexTeamBackendContext.objects\
                .filter(maillist=maillist).update(fsuid=fsuid, uid=uid)
            return True
        else:
            logger.error("GET_SUID returned none for %s", backend_context.passport_name)
            return False


# instantiated interface ready to use
backend_interface = MaillistBackendInterface()
