# -*- coding: utf-8 -*-
import logging
from operator import attrgetter

from django.conf import settings
from django.utils.encoding import smart_unicode
from django.dispatch import Signal
from django.contrib.auth.models import User

from django_intranet_stuff.utils import json
from mlcore.utils.encoding import deutify_keys
from mlcore.ml.models import EmailSubscriber, Subscribers,\
         PendingSubscribeOperation, PendingBackendOperation
from mlcore.stats.models import EmailActivity, SubscribeActivity
from mlcore.subscribe.backends.utils import operation_context_to_backend_method

logger = logging.getLogger(__name__)

operation_success = Signal(providing_args=['instance'])

class Operation(object):
    '''
    Base class for maillist operations: subscription and unsubscription
    '''

    id = None

    def __init__(self, list=None, **kwargs):
        self.list = list

    def isPossible(self):
        raise NotImplementedError

    def getErrors(self):
        raise NotImplementedError

    def perform(self):
        raise NotImplementedError

    def getSuccessMessage(self):
        raise NotImplementedError


class BackendOperation(Operation):

    def get_base_operation_context(self):
        return {}

    def get_backend_contexts(self):
        return (b.get_clarified_context()
            for b in self.list.backends.only('id', 'backend_type'))

    @classmethod
    def get_backend_operation(cls, backend, op_context):
        method_name = operation_context_to_backend_method(cls.id, op_context)
        backend_method = getattr(backend, method_name)
        return backend_method

    def _perform_for_backend(self, backend, op_context):
        backend_op = self.get_backend_operation(backend, op_context)
        result = backend_op(**op_context)
        if not result:
            raise Exception("Backend method returned False")

    def perform(self, backend_context):
        backend = backend_context.backend
        op_context = self.get_base_operation_context()
        op_context['backend_context'] = backend_context

        try:
            self._perform_for_backend(backend, op_context)
        except:
            logger.exception("Operation with id %s failed on backend %s:",
                    self.id, backend_context.backend_type)
            return False
        else:
            return True


class DeferrableOperation(BackendOperation):

    always_deferred_backends = ()

    def __init__(self, list=None):
        super(DeferrableOperation, self).__init__(list=list)

    @classmethod
    def serialize_operation_context(cls, op_context):
        backend_ctx = op_context['backend_context']
        op_context['backend_context'] = backend_ctx.as_dict()
        dumped_op_context = json.dumps(op_context)
        op_context['backend_context'] = backend_ctx
        return dumped_op_context

    def _save_pending_operation(self, backend_type,
            operation_id, operation_args, last_failure_info=''):
        # convert BackendContext instance to serializable dict
        dumped_operation_args = self.serialize_operation_context(operation_args)
        PendingBackendOperation.objects.create(
            backend_type=backend_type,
            operation_type=operation_id,
            operation_args=dumped_operation_args,
            last_failure_info=last_failure_info,
            max_attempts=settings.PENDING_OPERATION_MAX_ATTEMPTS)

    def perform_deferrable(self, backend, context):
        op_context = self.get_base_operation_context()
        op_context['backend_context'] = context
        try:
            self._perform_for_backend(backend, op_context)
        except BaseException, e:
            logger.exception("Operation with id %s failed on backend %s, but we postponed it", self.id, backend.type)
            self._save_pending_operation(backend.type, self.id, op_context,
                    last_failure_info=smart_unicode(e)[:255])
            return False
        else:
            return True

    def _patch_context(self, backend_context):
        '''
        Set backend_context.maillist attribute to avoid extra query
        (as we always have maillist instance in operation context)
        '''
        assert self.list
        backend_context.maillist = self.list
        return backend_context

    def get_sorted_backend_contexts(self):
        return sorted(
                map(self._patch_context, self.get_backend_contexts()),
                key=attrgetter('is_master_backend'), reverse=True)

    @staticmethod
    def omit_master_contexts(contexts):
        iterator = iter(contexts)
        first = None
        for first in iterator:
            if not first.is_master_backend:
                break
        if first is not None and first.is_master_backend:
            first = None
        return first, iterator

    def perform(self, force_pending=False, omit_master=False):

        backend_contexts = iter(self.get_sorted_backend_contexts())

        if not omit_master:

            if force_pending:
                for context in backend_contexts:
                    self.perform_deferrable(context.backend, context)
            else:
                try:
                    first = backend_contexts.next()
                except StopIteration:
                    pass
                else:
                    if first.is_master_backend:
                        required_result = super(DeferrableOperation, self).perform(first)
                        if not required_result:
                            return False
                    else:
                        self.perform_deferrable(first.backend, first)
        else:
            first, backend_contexts = self.omit_master_contexts(backend_contexts)
            if first is not None:
                self.perform_deferrable(first.backend, first)

        for context in backend_contexts:
            self.perform_deferrable(context.backend, context)

        return True


class UserOperation(DeferrableOperation):
    '''
    Base class for user-based subscriptions (user from staff subscribes)
    '''

    def __init__(self, user=None, list=None, context=None):
        super(UserOperation, self).__init__(list=list)
        self.user = user
        self.context = context if context is not None else {}
        self.autosubscription = None

    def _save_pending_operation(self,
            backend_type, operation_id, operation_args, last_failure_info=''):
        operation_args['backend_context'] = operation_args['backend_context'].as_dict()
        dumped_operation_args = json.dumps(operation_args)
        PendingSubscribeOperation.objects.create(
            user=self.user,
            list=self.list,
            backend_type=backend_type,
            operation_type=operation_id,
            operation_args=dumped_operation_args,
            last_failure_info=last_failure_info,
            max_attempts=settings.PENDING_OPERATION_MAX_ATTEMPTS)

    def get_base_operation_context(self):
        return dict(user=self.user)

    def perform(self, autosubscription=None, **kwargs):
        self.autosubscription = autosubscription
        result = super(UserOperation, self).perform(**kwargs)
        if result:
            operation_success.send_robust(UserOperation, instance=self)
        return result


class EmailOperation(DeferrableOperation):
    '''
    Base class for email-based subscription/unsubscription
    '''

    def __init__(self, list=None, email=None):
        super(EmailOperation, self).__init__(list=list)
        self.email = email

    def get_base_operation_context(self):
        return dict(email=self.email)

    def get_backend_contexts(self):
        return (b.get_clarified_context()\
            for b in self.list.backends.only('id', 'backend_type')
                                       .filter(is_sub=True))

    def perform(self, **kwargs):
        result = super(EmailOperation, self).perform(**kwargs)
        if result:
            operation_success.send_robust(EmailOperation, instance=self)
        return result


def save_user_subscription(sender, **kwargs):
    instance = kwargs['instance']
    try:
        s = Subscribers.objects.get(user=instance.user, list=instance.list)
    except Subscribers.DoesNotExist:
        s = Subscribers(user=instance.user, list=instance.list)

    if instance.id == "+imap":
        s.is_imap = True
    elif instance.id == "+inbox":
        s.is_sub = True
    elif instance.id == "-imap":
        s.is_imap = False
    elif instance.id == "-inbox":
        s.is_sub = False

    s.autosubscription = instance.autosubscription

    if not s.is_imap and not s.is_sub:
        s.delete()
    else:
        s.save()

operation_success.connect(save_user_subscription, sender=UserOperation)


def log_user_operation(sender, **kwargs):
    instance = kwargs['instance']
    s = SubscribeActivity(list=instance.list, user=instance.user,
            where=SubscribeActivity.WHERES[1][0])
    if instance.id == "+imap":
        s.sub_action = 1 # subscribe
        s.sub_type = 1 # imap
    elif instance.id == "+inbox":
        s.sub_action = 1 # subscribe
        s.sub_type = 2 # inbox
    elif instance.id == "-imap":
        s.sub_action = 0 # unsubscribe
        s.sub_type = 1 # imap
    elif instance.id == "-inbox":
        s.sub_action = 0 # unsubscribe
        s.sub_type = 2 # inbox
    s.save()

operation_success.connect(log_user_operation, sender=UserOperation)


def save_email_subscription(sender, **kwargs):
    instance = kwargs['instance']
    try:
        s = EmailSubscriber.objects.get(email=instance.email, list=instance.list)
        if instance.id == "-email":
            s.delete()
    except EmailSubscriber.DoesNotExist:
        s = EmailSubscriber(email=instance.email, list=instance.list)
        if instance.id == "+email":
            s.save()
    instance.log_operation()

operation_success.connect(save_email_subscription, sender=EmailOperation)


def log_email_operation(sender, **kwargs):
    instance = kwargs['instance']
    s = EmailActivity(list=instance.list, email=instance.email, operation=instance.id)
    s.save()

operation_success.connect(log_email_operation, sender=EmailOperation)
