# -*- coding: utf-8 -*-
from django.utils.translation import ugettext_lazy as _

from mlcore.subscribe.signals import operation_list_performed

class SubscriptionOperationListType:
    SUBSCRIBE = 'subscribe'
    UNSUBSCRIBE = 'unsubscribe'
    CHANGE_SUBSCRIPTION_TYPE = 'change_type' # both subscribe and unsubscribe
    types = (SUBSCRIBE, UNSUBSCRIBE, CHANGE_SUBSCRIPTION_TYPE)
    choices = (
        (SUBSCRIBE, _('subscription')),
        (UNSUBSCRIBE, _('unsubscription')),
        (CHANGE_SUBSCRIPTION_TYPE, _('subscription type change')),
        )

class OperationList(object):

    _type = None

    def __init__(self, operations):
        self.operations = operations
        super(OperationList, self).__init__()

    def __repr__(self):
        return "<%s: [%s]>" % (self.__class__.__name__,
                ", ".join(op.__class__.__name__ for op in self.operations))

    def setUser(self, user):
        for op in self.operations:
            op.user = user
        self.user = user

    def getUser(self):
        if self.isEmpty():
            return None

        op = self.operations[0]
        return op.user if hasattr(op, 'user') else None

    def setList(self, list):
        for op in self.operations:
            op.list = list
        self.list = list

    def getList(self):
        for op in self.operations:
            return op.list
        return None

    def setContext(self, context):
        for op in self.operations:
            op.context = context

    def getContext(self):
        context = {}
        for op in self.operations:
            context.update(op.context)
        return context

    def isEmpty(self):
        return len(self.operations) == 0

    def isPossible(self):
        possible = True
        for op in self.operations:
            if not op.isPossible():
                possible = False
                break

        if not self._isPossible():
            possible = False
        return possible

    def getErrors(self):
        errors = []
        for op in self.operations:
            errors += op.getErrors()
        return errors + self._getErrors()

    def getSuccessMessage(self):
        message = ""
        for op in self.operations:
            message += op.getSuccessMessage() + "\n"
        return message

    def getType(self):
        return self._type

    def perform(self, **kwargs):
        okay = True
        for op in self.operations:
            result = op.perform(**kwargs)
            if not result:
                okay = False
                # really: break?
                break
        operation_list_performed.send(
                sender=self.__class__, instance=self, result=okay
                )
        return okay

    def _isPossible(self):
        return True

    def _getErrors(self):
        return []


class SubscriptionOperationList(OperationList):
    def __init__(self, operations, op_list_type=None, from_type=None, to_type=None):
        self.from_type = from_type
        self.to_type = to_type
        self._type = op_list_type
        super(SubscriptionOperationList, self).__init__(operations=operations)

    def _isPossible(self):
        return not self.list.is_mandatory or (self._type != SubscriptionOperationListType.UNSUBSCRIBE)

    def _getErrors(self):
        if self.list.is_mandatory and (self._type == SubscriptionOperationListType.UNSUBSCRIBE):
            return [_("You cannot unsubscribe from this mailing list")]
        return []
