# -*- coding: utf-8 -*-
import logging

from django.utils.translation import ugettext as _

from mlcore.ml.notifications import SubscribedByFriendEmail, SubscribedAutomatically

from mlcore.subscribe.operations.base import UserOperation, EmailOperation, DeferrableOperation, operation_success
from mlcore.subscribe.operations.factory import register_operation

logging = logging.getLogger(__name__)

class AskPermission(UserOperation):

    def send_notification(self):
        from mlcore.permissions.notifications import AskPermission as N
        m = N(self)
        m.send()

    def save_request(self):
        from mlcore.permissions.models import ListPermission, Type
        try:
            ListPermission.objects.get(user=self.user, list=self.list, type__name="read")
        except ListPermission.DoesNotExist:
            p = ListPermission(user=self.user, list=self.list, type=Type.objects.get(name="read"))
            p.save()

    def perform(self, **kwargs):
        self.save_request()
        self.send_notification()
        return True

    def isPossible(self):
        return True

    def getErrors(self):
        return []

    def getSuccessMessage(self):
        return _(u"MSG.REQUEST_FOR_CLOSE_LIST_IS_SEND %(list_name)s") % {'list_name':self.list.name}


class AskPermissionWithSubscription(UserOperation):

    def __init__(self, opList):
        super(AskPermissionWithSubscription, self).__init__()
        self.user = opList.getUser()
        self.list = opList.getList()
        self.type = None

    def send_notification(self):
        from mlcore.permissions.notifications import AskPermissionWithSubscription as N
        m = N(self)
        m.send()

    def save_request(self):
        from mlcore.permissions.models import ListPermission, SubscriptionRequest, Type
        try:
            ListPermission.objects.get(user=self.user, list=self.list, type__name="read")
        except ListPermission.DoesNotExist:
            p = ListPermission(user=self.user, list=self.list, type=Type.objects.get(name="read"))
            p.save()
            r = SubscriptionRequest(permission=p, type=self.type)
            r.save()

    def perform(self, **kwargs):
        self.save_request()
        self.send_notification()
        return True

    def isPossible(self):
        return True

    def getErrors(self):
        return []

    def getSuccessMessage(self):
        return _(u"MSG.REQUEST_FOR_CLOSE_LIST_IS_SEND %(list_name)s") % {'list_name':self.list.name}


class AskPermissionForFriend(AskPermissionWithSubscription):

    def __init__(self, opList, meddler):
        super(AskPermissionForFriend, self).__init__(opList)
        self.meddler = meddler

    def send_notification(self):
        from mlcore.permissions.notifications import AskPermissionForFriend as N
        m = N(self)
        m.send()

    def save_request(self):
        from mlcore.permissions.models import ListPermission, SubscriptionRequest, Type
        try:
            ListPermission.objects.get(user=self.user, list=self.list, type__name="read")
        except ListPermission.DoesNotExist:
            p = ListPermission(user=self.user, list=self.list, type=Type.objects.get(name="read"))
            p.save()
            r = SubscriptionRequest(permission=p, type=self.type, meddler=self.meddler)
            r.save()

    def getSuccessMessage(self):
        return _(u"MSG.REQUEST_FOR_CLOSE_LIST_IS_SEND %(list_name)s %(username)s") % {'list_name':self.list.name, 'username' : self.user.username}


class InboxOperation(UserOperation):

    def get_backend_contexts(self):
        return (b.get_clarified_context()
            for b in self.list.backends.only('id', 'backend_type')
                                       .filter(is_sub=True))


class InboxSubscribe(InboxOperation):

    id = "+inbox"

    def isPossible(self):
        if not self.list.is_sub:
            return False
        return True

    def getErrors(self):
        errors = []
        if not self.list.is_sub:
            errors.append(_(u"MSG.NOT_INBOX_LIST %(list_name)s") % {'list_name' : self.list.name})
        return errors

    def getSuccessMessage(self):
        if 'meddler' in self.context and self.context['meddler']:
            return _(u"MSG.YOU_SUBSCRIBED_IN_INBOX %(username)s %(list_name)s") % {'username' : self.user, 'list_name' : self.list.name}

        return _(u"MSG.YOU_ARE_SUBSCRIBED_IN_INBOX %(list_name)s") % {'list_name' : self.list.name}

register_operation(InboxSubscribe)


class InboxUnsubscribe(InboxOperation):

    id = "-inbox"

    def isPossible(self):
        if not self.list.is_sub:
            return False
        return True

    def getErrors(self):
        errors = []
        if not self.list.is_sub:
            errors.append(_(u"MSG.NOT_INBOX_LIST %(list_name)s") % {'list_name' : self.list.name})
        return errors

    def getSuccessMessage(self):
        return _(u"MSG.YOU_UNSIBSCIBED_FROM_INBOX_LIST %(list_name)s") % {'list_name' : self.list.name}

register_operation(InboxUnsubscribe)


class ImapOperation(UserOperation):

    def get_backend_contexts(self):
        return (b.get_clarified_context()\
            for b in self.list.backends.only('id', 'backend_type')
                                       .filter(is_imap=True))


class ImapSubscribe(ImapOperation):

    id = "+imap"

    def isPossible(self):
        if not self.list.is_imap:
            return False
        return True

    def getErrors(self):
        errors = []
        if not self.list.is_imap:
            errors.append(_(u"MSG.LIST_NOT_FOR_IMAP_READING"))
        return errors

    def getSuccessMessage(self):
        if 'meddler' in self.context and self.context['meddler']:
            return _(u"MSG.YOU_SUBSCRIBED_IN_IMAP %(username)s %(list_name)s") % {'username' : self.user, 'list_name' : self.list.name}

        return _(u"MSG.YOU_ARE_SUBSCRIBED_IN_IMAP %(list_name)s") % {'list_name' : self.list.name}

register_operation(ImapSubscribe)


class ImapUnsubscribe(ImapOperation):

    id = "-imap"

    def isPossible(self):
        if not self.list.is_imap:
            return False
        return True

    def getErrors(self):
        errors = []
        if not self.list.is_imap:
            errors.append(_(u"MSG.LIST_NOT_FOR_IMAP_READING"))
        return errors

    def getSuccessMessage(self):
        return _(u"MSG.YOU_UNSIBSCRIBED_FROM_IMAP")

register_operation(ImapUnsubscribe)


class EmailSubscribe(EmailOperation):

    id = "+email"

    def isPossible(self):
        if not self.list.is_sub:
            return False
        return True

    def getErrors(self):
        errors = []
        if not self.list.is_sub:
            errors.append(_(u"MSG.NOT_INBOX_LIST %(list_name)s") % {'list_name' : self.list.name})
        return errors

    def getSuccessMessage(self):
        return _(u"MSG.SUBSCRIBED_TO_LIST %(email)s %(list_name)s") % {'email' : self.email, 'list_name' : self.list.name}

register_operation(EmailSubscribe)


class EmailUnsubscribe(EmailOperation):

    id = "-email"

    def isPossible(self):
        if not self.list.is_sub:
            return False
        return True

    def getErrors(self):
        errors = []
        if not self.list.is_sub:
            errors.append(_(u"MSG.NOT_INBOX_LIST %(list_name)s") % {'list_name' : self.list.name})
        return errors

    def getSuccessMessage(self):
        return _(u"MSG.UNSUBSCRIBED_TO_LIST %(email)s %(list_name)s") % {'email' : self.email, 'list_name' : self.list.name}

register_operation(EmailUnsubscribe)


class CreateMaillist(DeferrableOperation):

    id = 'create_maillist'

    def get_backend_contexts(self):
        return filter(lambda ctx: not ctx.is_master_backend,
            (b.get_clarified_context()
                for b in self.list.backends.only('id', 'backend_type')))

register_operation(CreateMaillist)


def send_subscription_notification(sender, **kwargs):

    instance = kwargs['instance']
    if isinstance(instance, (ImapSubscribe, InboxSubscribe)):
        meddler = instance.context['meddler']

        to = [instance.user.email]
        cc = [meddler.email]

        SubscribedByFriendEmail({
            'list': instance.list,
            'meddler': meddler,
        }).send(to, cc=cc)


operation_success.connect(send_subscription_notification, sender=UserOperation)

def send_autosubscription_notification(sender, **kwargs):
    instance = kwargs['instance']

    if instance.autosubscription is not None and "+" in instance.id:
        to = [instance.user.email]
        if instance.id == '+imap':
            stype = _('AUTOSUBSCRIBE.TO_SHARED_FOLDER')
        elif instance.id == '+inbox':
            stype = _('AUTOSUBSCRIBE.TO_INBOX')
        SubscribedAutomatically({
            'list': instance.list,
            'owners': [owner.user.get_profile()
                       for owner in instance.list.owner_set.all()],
            'stype': stype,
        }).send(to)

operation_success.connect(send_autosubscription_notification)

