# -*- coding: utf-8 -*-

from mlcore.subscribe import subscription_type
from mlcore.subscribe.operations.factory import get_operation
from mlcore.subscribe.operations.list import SubscriptionOperationList,\
         SubscriptionOperationListType

__all__ = ['get_transitional_operations', 'get_type_changer']

class _SubscriptionType(object):
    def __init__(self, components, type_):
        self.components = components
        self._type = type_

    def getOperations(self, other_type):
        to_remove = self.components.difference(other_type.components)
        to_add = other_type.components.difference(self.components)
        ids = []
        for i in to_remove:
            ids.append("-" + i)
        for i in to_add:
            ids.append("+" + i)

        operations = []
        for id in ids:
            operations.append(get_operation(id))

        op_list_type = None
        if to_add and not to_remove:
            op_list_type = SubscriptionOperationListType.SUBSCRIBE
        elif to_add and to_remove:
            op_list_type = SubscriptionOperationListType.CHANGE_SUBSCRIPTION_TYPE
        elif not to_add and to_remove:
            if not other_type.components:
                op_list_type = SubscriptionOperationListType.UNSUBSCRIBE
            else:
                op_list_type = SubscriptionOperationListType.CHANGE_SUBSCRIPTION_TYPE

        return SubscriptionOperationList(operations, op_list_type,
                from_type=self._type, to_type=other_type._type)


TYPE_CHANGER_MAPPING = {
    subscription_type.NONE:
        _SubscriptionType(set(), subscription_type.NONE),
    subscription_type.EMAIL:
        _SubscriptionType(set([subscription_type.EMAIL]), subscription_type.EMAIL),
    subscription_type.IMAP:
        _SubscriptionType(set([subscription_type.IMAP]), subscription_type.IMAP),
    subscription_type.INBOX:
        _SubscriptionType(set([subscription_type.INBOX]), subscription_type.INBOX),
    subscription_type.BOTH:
        _SubscriptionType(set(
                [subscription_type.IMAP,
                subscription_type.INBOX]
                ), subscription_type.BOTH),
}


def get_type_changer(type_):
    '''
    Return type changer instance from type code.
    '''
    type_changer = TYPE_CHANGER_MAPPING.get(type_)
    if type_changer is None:
        raise TypeError(type_)
    return type_changer


def get_transitional_operations(old_composite_type, new_composite_type):
    '''
    For two composite subscription types returns operation list necessary to
    perform transition from old type to new.

    Will raise TypeError if any of two types are unknown.

    Example (assuming that "both" and "email" are defined types in TYPE_CHANGER_MAPPING):

    >>> get_transitional_operations("both", "email")
    <SubscriptionOperationList: [ImapUnsubscribe, InboxUnsubscribe, EmailSubscribe]>
    '''
    old = get_type_changer(old_composite_type)
    new = get_type_changer(new_composite_type)
    return old.getOperations(new)
