# coding: utf-8

"""Базовыми являются операции которые уже кое-что знают о специфике
сервиса. Например то что у нас есть ymail и к кому обращаться
по какому вопросу. Этот модуль реализовывает базовые возможности
сервиса (подписка во входящие, отписка от shared folder), приэтом
ничего не знает о правах доступа и всей такой прочей бизнес-логике.

ВАЖНО! Все операции начинаем с ML, а в конце ymail, так как они зависят от прав,
которые записаны у нас.

Базовые операции должны быть построены на low_level операциях, если
это почему-то не так, то нужно помнить что для всех операций должно
выполняться требование идемпотентности.
"""

import logging

from django.core.exceptions import ObjectDoesNotExist
from django.db import transaction
from django.conf import settings
from copy import deepcopy

from celery import task, chain

from mlcore.ml.notifications import SubscribedByFriendEmail, MailListCreated
from mlcore.subscribe import subscription_type as sub_type
from mlcore.utils.getters import get_list, get_user
from mlcore.tasks.log import MLTask
from low_level import (subscribe_ymail, subscribe_ml,
                       unsubscribe_ymail, unsubscribe_ml,
                       create_passport_user,
                       create_maillist_ml, create_maillist_ymail,
                       rnd_delay,
                       pdd_create_maillist, passport_set_altdomain,
                       check_yateam_blackbox_user, create_separate_folder_and_filter,
                       move_user_to_pg,
                       subscribe_york,
                       unsubscribe_york,
                       create_maillist_york)
from mlcore.tasks.notifications import NewSubscribe
from mlcore.ml.models import MailList, Owner, CorporateDomain, DirectForwards, EmailSubscriber, Subscribers

from mlcore.ml.exceptions import MailListNotCreate
from mlcore.pg_migration.utils import get_replica_login, get_replica_email, user_is_pg


def create_request_context(submitted_data, staff):
    '''
    Create context for create request email based on submitted form data
    submitted_data = {'info': u'qwe',
                      'info_en': u'qwe',
                      'is_internal': True,
                      'is_open': True,
                      'name': u'qweqwe123123@yandex-team.ru',
                      'responsible': set([u'zubchick@yandex-team.ru'])}
    return = {'domain': u'yandex-team.ru',
              'email': u'qweqwe123123@yandex-team.ru',
              'imap_name': u'qweqwe123123',
              'info': u'qwe',
              'info_en': u'qwe',
              'is_internal': True,
              'is_open': True,
              'name': u'qweqwe123123',
              'omit_yandex_team': False,
              'responsible': set([u'zubchick@yandex-team.ru'])}
    '''
    new_list = submitted_data.copy()
    new_list['email'] = submitted_data['name']
    new_list['name'], new_list['domain'] = new_list['email'].split('@')
    new_list['omit_yandex_team'] = False
    domain = new_list['domain']

    if not new_list['responsible']:
        new_list['responsible'] = staff.email

    if domain in settings.YANDEX_TEAM_DOMAINS:
        new_list['imap_name'] = new_list['name'].replace('.', '-')
    else:
        domain_suffix = domain.replace('.', '-')
        new_list['imap_name'] = u'-at-'.join(
            [new_list['name'].replace('.', '-'), domain_suffix])

    return new_list


def subscribe_inbox(context, user, maillist, autosubscription=None, send_notification=False, separate=False):
    """ Подписка в inbox """
    maillist = get_list(maillist)

    if not maillist.is_sub:
        return finish_task.si(context,
                              u"В рассылку %s нельзя подписаться в inbox",
                              logging.WARNING, maillist.name)
    subscribe_chain = subscribe_ml.si(context, user, maillist, sub_type.INBOX if not separate else sub_type.SEPARATE,
                                      autosubscription=autosubscription)

    if separate:
        subscribe_chain |= create_separate_folder_and_filter.si(context, user, maillist)

    if send_notification:
        subscribe_chain |= notify_about_subscribe.si(context, user,
                                                     maillist, sub_type.INBOX if not separate else sub_type.SEPARATE)

    return subscribe_chain


def unsubscribe_inbox(context, user, maillist):
    """ Отписка от inbox """
    maillist = get_list(maillist)
    if not maillist.is_sub:
        return finish_task.si(context,
                              u"В рассылку %s нельзя подписаться в inbox",
                              logging.WARNING, maillist.name)
    return unsubscribe_ml.si(context, user, maillist, sub_type.INBOX)


def subscribe_imap(context, user, maillist, autosubscription=None, send_notification=False):
    """ Подписка в shared folder """
    maillist = get_list(maillist)
    if not maillist.is_imap:
        return finish_task.si(context,
                              u"В рассылку %s нельзя подписаться в shared folder",
                              logging.WARNING, maillist.name)

    subscribe_backend_task = subscribe_ymail
    if settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_ORA_PG:
        if user_is_pg(user):
            subscribe_backend_task = subscribe_york
    elif settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_PG:
        # TODO: после перезеда в PG оставить только эту ветку
        subscribe_backend_task = subscribe_york

    subscribe_chain = (subscribe_ml.si(context, user, maillist, sub_type.IMAP,
                                       autosubscription=autosubscription) |
                       subscribe_backend_task.si(context, user, maillist, sub_type.IMAP,
                                                 autosubscription=autosubscription))
    if send_notification:
        subscribe_chain |= notify_about_subscribe.si(context, user, maillist, sub_type.IMAP)

    return subscribe_chain


def unsubscribe_imap(context, user, maillist, autosubscription=None):
    """ Отписка из shared folder """
    maillist = get_list(maillist)
    if not maillist.is_imap:
        return finish_task.si(context,
                              u"В рассылку %s нельзя подписаться в shared folder",
                              logging.WARNING, maillist.name)

    unsubscribe_backend_task = unsubscribe_ymail
    if settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_ORA_PG:
        if user_is_pg(user):
            unsubscribe_backend_task = unsubscribe_york
    elif settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_PG:
        # TODO: после перезеда в PG оставить только эту ветку
        unsubscribe_backend_task = unsubscribe_york

    return (unsubscribe_ml.si(context, user, maillist, sub_type.IMAP) |
            unsubscribe_backend_task.si(context, user, maillist, sub_type.IMAP, autosubscription=autosubscription))


def subscribe_both(context, user, maillist, autosubscription=None, send_notification=False):
    """ Подписка в inbox и imap"""
    subscribe_backend_task = subscribe_ymail
    if settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_ORA_PG:
        if user_is_pg(user):
            subscribe_backend_task = subscribe_york
    elif settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_PG:
        # TODO: после перезеда в PG оставить только эту ветку
        subscribe_backend_task = subscribe_york

    subscribe_chain = (
        subscribe_ml.si(context, user, maillist, sub_type.BOTH, autosubscription=autosubscription) |
        subscribe_backend_task.si(context, user, maillist, sub_type.IMAP, autosubscription=autosubscription)
    )

    if send_notification:
        subscribe_chain |= notify_about_subscribe.si(context, user, maillist, sub_type.BOTH)

    return subscribe_chain


def unsubscribe_both(context, user, maillist, autosubscription=None):
    """ Отписка из inbox и imap """
    return (unsubscribe_inbox(context, user, maillist) |
            unsubscribe_imap(context, user, maillist, autosubscription=autosubscription))


def create_maillist(context, data, user, use_cmail=False):
    """
    Создание рассылки, учитывая MX-ы.
    ML-1485: в самом начале проверяем наличие аккаунта с таким email-ом в yateam ЧЯ.
    Если должно быть создано через ПДД-схему, то тоже проверим, есть ли такой аккаунт в Большом ЧЯ
    (Чтоб ПДД ручку лишний раз не дергать)
    """
    full_data = create_request_context(data, user)

    immutable_subtasks = [check_yateam_blackbox_user.si(context, full_data)]

    # TODO: выпилить после перезда в PG
    if settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_ORA_PG:
        replica_data = {
            'email': get_replica_email(full_data['imap_name']),
            'imap_name': get_replica_login(full_data['imap_name'])
        }
        immutable_subtasks += [
            check_yateam_blackbox_user.si(context, replica_data),
            create_passport_user.si(context, replica_data),
            move_user_to_pg.si(context, replica_data)
        ]

    immutable_subtasks += [create_passport_user.si(context, full_data),
                           create_maillist_ml.si(context, full_data), ]

    # TODO: выпилить после перезда в PG
    if settings.PG_MIGRATION_STATE != settings.PG_MIGRATION_STATE_PG:
        immutable_subtasks.append(create_maillist_ymail.si(context, full_data))

    if settings.PG_MIGRATION_STATE != settings.PG_MIGRATION_STATE_ORA:
        immutable_subtasks.append(create_maillist_york.si(context, full_data))

    domain = CorporateDomain.objects.get(domain=full_data['domain'])

    if domain.is_pdd and domain.domain not in settings.PASSPORT_ALIASES and domain.domain not in settings.YANDEX_TEAM_DOMAINS:
        # ML-1466: Завести рассылку в ПДД
        data = dict(domain=full_data['domain'], name=full_data['name'],
                    imap_email=full_data['imap_name'] + '@yandex-team.ru')
        immutable_subtasks = [check_yateam_blackbox_user.si(context, full_data),
                              pdd_create_maillist.si(context, **data)] + immutable_subtasks

    if domain.is_type9:
        # Прописать паспортный алиас
        immutable_subtasks += [passport_set_altdomain.si(context, full_data['imap_name'], full_data['email'])]

    if user:
        immutable_subtasks.append(notify_about_maillist_creation.si(context, full_data, user))
    return chain(s for s in immutable_subtasks)


def create_maillist_as_inbox_email(context, full_data, full_data_ml, user):
    """ Создание рассылки в виде юзера с email-ом без хождения в cmail."""
    create_maillist_chain = (create_passport_user.si(context, full_data) |
                             create_maillist_ml.si(context, full_data_ml))
    return create_maillist_chain


@task(base=MLTask)
def notify_about_maillist_creation(context, data, user):
    """ Уведомление о том, что рассылка была созадана ML-1553
    """
    try:
        maillist = MailList.objects.get(name=data['name'] if data['domain'] == u'yandex-team.ru' else data['email'])
        notification = MailListCreated(context={'list': maillist})
        notification.send([user.email])
    except MailList.DoesNotExist:
        logging.warning("Mail list '%s' does not exist", data['name'])


@task(base=MLTask)
def finish_task(context, message, level, *args, **kwargs):
    """ Дополнительная информация

    Служебная таска.
    Нужна чтобы не возвращать None из базовой таски и писать логи :)
    """
    logger = finish_task.request.logger
    logger.log(level, message, *args, **kwargs)


@task(base=MLTask)
def notify_about_subscribe(context, meddler, maillist, sub_type=None):
    """ Уведомление подписавшегося о новой подписке

    meddler - тот, кто подписался
    maillist - на этот список подписался
    """

    logger = notify_about_subscribe.request.logger
    user = get_user(meddler)
    maillist = get_list(maillist)

    subscription = Subscribers.objects.filter(user=user, list=maillist, stype=sub_type)
    if not subscription and sub_type:
        logger.warning(u'Пользователь не был подписан, поэтому пропускаю отправку нотификации')
        return

    try:
        initiator = get_user(context.get('initiator', user))
    except ObjectDoesNotExist:
        initiator = None

    logger.info(u"Отсылаю письмо с уведомлением о новой подписке на %s", user.email)
    if initiator and initiator != user:
        # Если подписал друг
        to = [user.email]
        cc = [initiator.email]

        SubscribedByFriendEmail({
            'list': maillist,
            'meddler': initiator,
        }).send(to, cc=cc)
    else:
        mail = NewSubscribe(maillist=maillist, user=user)
        mail.send([user.email])


def create_alias(context, email, owner):
    """
    Создание алиаса в качестве рассылки.
    """
    name, domain = email.split('@')
    custom_email = name.replace('.', '-') + '@' + domain
    data = {
        'name': custom_email,
        'is_open': False, 'is_internal': False,
        'info': email, 'info_en': email,
        'before_clean': 0,
        'archive': False,
        'responsible': set([owner.email], )
    }
    full_data = create_request_context(data, owner)
    context = 'Create maillists alias: %s' % email

    # формируем имя в ML с точками, если они есть
    name_ml = email.split('@')[0] if email.endswith('@yandex-team.ru') else email
    full_data_ml = deepcopy(full_data)
    full_data_ml['name'] = name_ml
    full_data_ml['email'] = email

    return create_maillist_as_inbox_email(context, full_data, full_data_ml, owner)


@task(base=MLTask, max_retries=42)
@transaction.atomic
def create_alias_and_subscribe_users(context, email, destinations, owners):
    logger = create_alias_and_subscribe_users.request.logger
    try:
        robot_owner = get_user('robot-yndx-maillists@yandex-team.ru')
        one_owner = owners[0] if owners else robot_owner
        create_alias(context, email, one_owner).delay()

        exists_maillist = MailList.objects.filter(email=email)
        if not exists_maillist:
            raise MailListNotCreate("Maillist with email %s not create yet" % email)

        maillist = exists_maillist[0]
        maillist.use_cmail = False
        maillist.is_imap = True
        maillist.is_sub = True
        maillist.save()

        for owner in owners:
            Owner.objects.get_or_create(user=owner, list=maillist)

        # подписать во входящие email-ы, если они есть
        emailsubscribers = set(destinations)
        chain_subscribe = []
        for email in emailsubscribers:
            chain_subscribe.append(subscribe_ml.si({
                'initiator': email,
                'comment': u'Подписка в рассылку-пользователя %s' % maillist.name
            }, user=email, maillist=maillist, type=sub_type.INBOX))
        chain(chain_subscribe).delay()

    except MailListNotCreate as exc:
        logger.exception(u"Maillist with email %s not create yet", email)
        raise create_alias_and_subscribe_users.retry(exc=exc, countdown=1 + rnd_delay(10))
    except Exception as exc:
        logger.exception(u"Unexpected error: %s", exc.message)
        raise create_alias_and_subscribe_users.retry(exc=exc, countdown=1 + rnd_delay(10))


def create_direct_forward(context, forward_id):
    try:
        fwd = DirectForwards.objects.get(id=forward_id, is_deleted=False)
    except DirectForwards.DoesNotExist:
        logging.error(u'Не найдено записи о пересылки с id %d', forward_id)
        return
    if fwd.is_deleted:
        logging.error(u'Пересылка на адрес %s удалена', fwd.email)
        return
    if fwd.status != DirectForwards.STATUS_APPROVED and not (
            fwd.status == DirectForwards.STATUS_ACTIVE and fwd.implemented is None):
        logging.error(u'Пересылка на адрес %s находится в неправильном статусе %s', fwd.email, fwd.status)
        return

    es, skip = EmailSubscriber.objects.get_or_create(list=fwd.maillist, email=fwd.email)

    fwd.implemented = es
    fwd.status = DirectForwards.STATUS_ACTIVE
    fwd.save()


def delete_direct_forward(context, forward_id):
    try:
        fwd = DirectForwards.objects.get(id=forward_id, is_deleted=False)
    except DirectForwards.DoesNotExist:
        logging.error(u'Не найдено записи о пересылки с id %d', forward_id)
        return
    if fwd.is_deleted:
        logging.error(u'Пересылка на адрес %s уже удалена', fwd.email)
        return

    es = fwd.implemented
    if es:
        fwd.implemented = None
        es.delete()
    fwd.is_deleted = True
    fwd.save()
