# coding: utf-8
import cPickle
from cStringIO import StringIO
import logging
from datetime import datetime
from celery import Task
from django.db import OperationalError
from mlcore.ml.exceptions import MailListNotCreate

from django_intranet_stuff.models import Staff
from django.contrib.auth.models import User
from mlcore.ml.models import OperationLog, MailList, Autosubscription
from mlcore.utils.getters import get_user, get_list


def dump_arguments(context, args, kwargs):
    data = {'context': context, 'args': [], 'kwargs': {}}

    def serialize_autosub(autosub):
        return (autosub.maillist.name, autosub.group_id, autosub.stype)

    def get_value(item):
        if isinstance(item, MailList):
            return item.name
        elif isinstance(item, User):
            return item.username
        elif isinstance(item, Staff):
            return item.login
        elif isinstance(item, Autosubscription):
            return serialize_autosub(item)
        else:
            return item

    for item in args:
        data['args'].append(get_value(item))

    for key, item in kwargs.iteritems():
        data['kwargs'][key] = get_value(item)

    return cPickle.dumps(data)


def init_task_logger(name):
    formatter = logging.Formatter(u'[%(process)d %(processName)s] [%(asctime)s] '
                                  '[%(levelname)s] %(message)s')

    stringio_handler = logging.StreamHandler(StringIO())
    stringio_handler.setFormatter(formatter)

    log = logging.getLogger(name)
    log.setLevel(logging.INFO)

    for handler in log.handlers[:]:
        log.removeHandler(handler)

    log.addHandler(stringio_handler)

    return log


class MLTask(Task):
    def __call__(self, context, *args, **kwargs):
        try:
            task_id = self.request.id
            task_name = self.name

            logger = init_task_logger(task_name)

            if context is None or not isinstance(context, dict):
                logger.warning(u"Контекст не был передан, возможно это ошибка")
                context = {}

            # Если таска в ручном ретрае у нее в контексе есть retry_parent
            retry_parent = context.pop('retry_parent', None)

            arguments = dump_arguments(context, args, kwargs)

            if not context.get('root_id'):
                context['root_id'] = task_id

            level = (self.name.rsplit('.', 1)[0]).replace('mlcore.tasks.', '')

            # При перезапуске рутового таска не ставить парента на
            # самого себя
            if task_id == context['root_id']:
                parent = None
            else:
                parent = context.get('current_id')

            oplog = OperationLog(
                root_id=context['root_id'], name=task_name, type=level,
                parent_id=parent, started=datetime.now(),
                comment=context.get('comment', ''), task_id=task_id,
                arguments=arguments, retry_parent_id=retry_parent)

            self.request.logger = logger
            self.request.oplog = oplog

            # заменяем id на текущий
            context['current_id'] = task_id

            logger.info(u"Операция %s запущена с параметрами args=%s, kwargs=%s",
                        task_name, args, kwargs)

            # попробуем достать дополнительную информацию из args
            users = filter(lambda x: isinstance(x, User), args)
            lists = filter(lambda x: isinstance(x, MailList), args)

            if users:
                oplog.user_id = users[0].id
            if lists:
                oplog.list_id = lists[0].id

            # попробуем достать дополнительную информацию из kwargs
            try:
                if 'user' in kwargs and kwargs['user']:
                    oplog.user = get_user(kwargs['user'])
                if 'maillist' in kwargs and kwargs['maillist']:
                    oplog.list = get_list(kwargs['maillist'])
                if 'meddler' in kwargs and kwargs['meddler']:
                    oplog.initiator = get_user(kwargs['meddler']).username
            except (User.DoesNotExist, MailList.DoesNotExist):
                logger.exception(u"Неизвестный пользователь или рассылка")

            if 'initiator' in context:
                oplog.initiator = context['initiator']

            # может что-то есть в контексте?
            if 'user' in context and context['user']:
                oplog.user = get_user(context.pop('user'))
            if 'list' in context and context['list']:
                oplog.list = get_list(context.pop('list'))

            # проверим можно ли ее ретраить
            if context.get('in_retry') and getattr(self, 'do_not_retry', False):
                if retry_parent:
                    parent = OperationLog.objects.get(task_id=retry_parent)
                    if parent.status == 'SUCCESS':
                        logger.info(u"Таск уже был успешно выполнен и не должен"
                                    u" перезапускаться")
                        oplog.status = 'SUCCESS'
                        oplog.messages = logger.handlers[0].stream.getvalue()
                        oplog.save()
                        return
                    else:
                        logger.info(u"Таск был завершен с ошибкой, поэтому"
                                    u" перезапускаю")
                else:
                    logger.info(u"Таск уже был выполнен и не должен "
                                u"перезапускаться")
                    oplog.status = 'SUCCESS'
                    oplog.messages = logger.handlers[0].stream.getvalue()
                    oplog.save()
                    return

            if task_id is None:
                logger.error(u"Таск %s запущен без .delay и не будет выполнен",
                             self.name)
                oplog.status = 'NOT-DELAYED'
                oplog.messages = logger.handlers[0].stream.getvalue()
                oplog.save()
                return

            oplog.status = 'PENDING'
            oplog.messages = logger.handlers[0].stream.getvalue()
            oplog.save()

            self.run(context, *args, **kwargs)

        except MailListNotCreate as exc:
            logger.exception("Рассылка еще не создалась")
            oplog.messages = logger.handlers[0].stream.getvalue()
            oplog.save()
            raise self.retry(exc=exc, countdown=30, max_retries=3)
        except OperationalError, exc:
            from .low_level import rnd_delay
            logger.exception(u"Ошибка базы данных")
            oplog.messages = logger.handlers[0].stream.getvalue()
            oplog.save()
            raise self.retry(exc=exc, countdown=1 + rnd_delay(5))
        except Exception:
            logger.exception(u"Непонятная ошибка при выполнении таска")
            oplog.messages = logger.handlers[0].stream.getvalue()
            oplog.save()
            raise

    def after_return(self, status, retval, task_id, args, kwargs, einfo):
        logger = self.request.logger
        oplog = self.request.oplog

        oplog.messages = logger.handlers[0].stream.getvalue()
        oplog.status = status
        oplog.finished = datetime.now()
        oplog.save()

    def on_retry(self, exc, task_id, args, kwargs, einfo):
        logger = self.request.logger
        oplog = self.request.oplog

        logger.error(u"Ошибка во время выполнения операции, пробуем еще раз",
                     exc_info=(einfo.type, einfo.exception, einfo.tb))
        oplog.status = 'RETRY'
        oplog.messages = logger.handlers[0].stream.getvalue()
        oplog.save()
