# coding: utf-8

"""Низкоуровневыми подразумеваются операции которые взаимодействуют
на прямую с бекендами через их протоколы. low_level операции ничего не
знают о том кто их вызывает и просто делают то что сказано

ВАЖНО! Low-level операции скрывают в себе реализации бекендов таким
образом чтобы выполнялось требование идемпотентности (например
подписать того же человека на ту же рассылку)

"""
from datetime import datetime
from random import gauss

from django.contrib.auth.models import User
from django.db import transaction
from django.conf import settings

from celery import task
from requests.exceptions import RequestException

from mlcore.interaction.ymail import YmailBackend, YmailError, PassportException
from mlcore.interaction.ywmi import YWMIBackend, YWMIError
from mlcore.interaction.pdd import PDDBackend, PDDError
from mlcore.interaction.mops import MopsBackend, MopsError
from mlcore.interaction.yfurita import SpecialMailListFolder, YFuritaError, YFuritaBackend
from mlcore.interaction.york import YorkBackend, YorkFatalError, YorkRepeatableError, YorkError, \
    YorkErrorWithNotification
from mlcore.subscribe import subscription_type as sub_type
from mlcore.utils.getters import get_user, get_list, get_suid, get_autosubscription

# models
from mlcore.ml.models import (Subscribers, EmailSubscriber, ARCHIVING_TYPE)
from mlcore.ml.notifications import YaMailSubscriptionProblem, YorkDuplicateFolderProblem
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext
from mlcore.permissions.models import ListPermission, Type
from mlcore.tasks.log import MLTask
from mlcore.utils.blackbox_util import get_uid_by_email, UserInBigBlackbox, UserInYaTeamBlackbox, get_uid, get_userinfo

from mlcore.ml.models import MailList, Owner
from ya_directory import DirectoryAPIError
from ya_directory import api as directory_api

dry_run = settings.DRY_RUN


def rnd_delay(seconds, shift=0.3):
    """ Случайное время задержки в пределах задаваемых shift
    max = seconds + shift * seconds  - максимально возможное значение
    min = seconds - shift * seconds  - минимальное возможное значение
    """
    res = -1
    max = seconds + shift * seconds
    min = seconds - shift * seconds
    while not min < res < max:
        res = gauss(seconds, seconds * shift * 0.5)

    return res


@task(base=MLTask)
def subscribe_ml(context, user, maillist, type, autosubscription=None):
    """ Подписка в ML """
    def _subscribe(context, user, maillist, type, autosubscription):
        assert type in (sub_type.IMAP, sub_type.INBOX, sub_type.BOTH, sub_type.SEPARATE)
        maillist = get_list(maillist)
        logger = subscribe_ml.request.logger

        if isinstance(user, basestring) and '@' in user:
            try:
                user = get_user(user)
            except User.DoesNotExist:
                logger.info(u"Пытаемся подписать в ML емейл %s", user)
                EmailSubscriber.objects.get_or_create(list=maillist, email=user)
                return
        else:
            user = get_user(user)

        autosub = get_autosubscription(autosubscription)

        try:
            sb = Subscribers.objects.get(list=maillist, user=user)
            logger.warning(u"Пользователь %s уже был подписан в ML в %s",
                           user, sb.stype)
        except Subscribers.DoesNotExist:
            sb = Subscribers(list=maillist, user=user, autosubscription=autosub, stype=type)

        if type == sub_type.IMAP:
            sb.is_imap = True
        elif type == sub_type.INBOX or type == sub_type.SEPARATE:
            sb.is_sub = True
        else:
            sb.is_sub = True
            sb.is_imap = True

        sb.stype = type
        sb.save()

    # Разбиваем на независимые транзакции
    with transaction.atomic():
        _subscribe(context, user, maillist, type, autosubscription)

    maillist.set_as_modified()


@task(base=MLTask)
def unsubscribe_ml(context, user, maillist, type):
    """ Отписка в ML """
    def _unsubscribe(context, user, maillist, type):
        assert type in (sub_type.IMAP, sub_type.INBOX)
        logger = unsubscribe_ml.request.logger
        maillist = get_list(maillist)

        if isinstance(user, basestring) and '@' in user:
            try:
                user = get_user(user)
            except User.DoesNotExist:
                logger.info(u"Отписываем в ML емейл %s", user)
                EmailSubscriber.objects.filter(list=maillist, email=user).delete()
                return
        else:
            user = get_user(user)

        if type == sub_type.IMAP:
            types = {'is_imap': True}
        else:
            types = {'is_sub': True}

        try:
            sb = Subscribers.objects.get(list=maillist, user=user, **types)
        except Subscribers.DoesNotExist:
            logger.warning(u"Нет такой подписки в ML")
            return
        if sb.stype == sub_type.BOTH:
            logger.info(u"Было %s отписываю в ML от %s", sb, type)
            sb.is_imap = not sub_type.IMAP == type
            sb.is_sub = not sub_type.INBOX == type
            if type == sub_type.IMAP:
                sb.stype = sub_type.INBOX
            else:
                sb.stype = sub_type.IMAP
            sb.save()
        elif sb.stype == type or (sb.stype == sub_type.SEPARATE and type == sub_type.INBOX):
            logger.info(u"Отписываю в ML %s", sb)
            sb.delete()

    # Разбиваем на независимые транзакции
    with transaction.atomic():
        _unsubscribe(context, user, maillist, type)

    maillist.set_as_modified()


@task(base=MLTask, max_retries=42)
def subscribe_ymail(context, user, maillist, stype, autosubscription=None):
    """ Подписка в Почте """
    assert stype in (sub_type.IMAP, )
    logger = subscribe_ymail.request.logger

    mdb = get_userinfo(user.staff.uid, by_login=False).get('fields', {}).get('mdb', settings.YWMI_API_MDB)

    ymail = YWMIBackend(logger=logger, dry_run=dry_run, mdb=mdb)

    try:
        try:
            ymail.subscribe(user, maillist)
        except (RequestException, IOError, ValueError), exc:
            logger.exception(u"Проблемы с mail.yandex-team.ru")
            raise subscribe_ymail.retry(exc=exc)
        except (YmailError, YWMIError) as exc:
            # Timed out while acquiring lock
            if exc.code == 100:
                raise subscribe_ymail.retry(exc=exc, countdown=rnd_delay(20))
            else:
                raise
    except (RequestException, IOError, ValueError):
        logger.error('Max retries to metacorp exceeded, remove subscription')
        roll_back_task = unsubscribe_ml.si(context, user, maillist, stype)
        if not autosubscription:
            roll_back_task |= notify_about_metacorp_problem.si(context, user, maillist)

        roll_back_task.delay()


@task(base=MLTask, max_retries=42)
def subscribe_york(context, user, maillist, stype, autosubscription=None):
    """ Подписка в Почте """
    assert stype in (sub_type.IMAP, )
    logger = subscribe_york.request.logger

    york = YorkBackend(logger=logger, dry_run=dry_run)

    try:
        try:
            york.subscribe(user, maillist)
        except (RequestException, IOError, ValueError), exc:
            logger.exception(u"Проблемы с York")
            raise subscribe_york.retry(exc=exc)
        except YorkRepeatableError as exc:
            raise subscribe_york.retry(exc=exc, countdown=rnd_delay(20))
        except YorkErrorWithNotification:
            roll_back_task = unsubscribe_ml.si(context, user, maillist, stype)
            roll_back_task |= notify_about_duplicate_folder_problem.si(context, user, maillist, autosubscription)
            roll_back_task.delay()
    except (RequestException, IOError, ValueError, YorkRepeatableError, YorkError):
        logger.exception('Failed to create subscription')
        roll_back_task = unsubscribe_ml.si(context, user, maillist, stype)
        if not autosubscription:
            roll_back_task |= notify_about_metacorp_problem.si(context, user, maillist)

        roll_back_task.delay()


@task(base=MLTask, max_retries=42)
def unsubscribe_ymail(context, user, maillist, stype, autosubscription=None):
    """ Отписка в Почте """
    assert stype in (sub_type.IMAP,)
    logger = unsubscribe_ymail.request.logger
    user = get_user(user)
    maillist = get_list(maillist)

    mdb = get_userinfo(user.staff.uid, by_login=False).get('fields', {}).get('mdb', settings.YWMI_API_MDB)

    ymail = YWMIBackend(logger=logger, dry_run=dry_run, mdb=mdb)

    try:
        try:
            ymail.unsubscribe(user, maillist)
        except (RequestException, IOError, ValueError), exc:
            logger.exception(u"Проблемы с mail.yandex-team.ru")
            raise unsubscribe_ymail.retry(exc=exc)
        except (YmailError, YWMIError), exc:
            # Timed out while acquiring lock
            if exc.code == 100:
                raise unsubscribe_ymail.retry(exc=exc, countdown=rnd_delay(20))
            else:
                raise
    except (RequestException, IOError, ValueError, YorkFatalError):
        logger.error('Max retries to metacorp exceeded, restore subscription')
        roll_back_task = subscribe_ml.delay(context, user, maillist, stype)
        if not autosubscription:
            roll_back_task |= notify_about_metacorp_problem.si(context, user, maillist, True)

        roll_back_task.delay()


@task(base=MLTask, max_retries=42)
def unsubscribe_york(context, user, maillist, stype, autosubscription=None):
    """ Отписка в Почте """
    assert stype in (sub_type.IMAP,)
    logger = unsubscribe_york.request.logger

    york = YorkBackend(logger=logger, dry_run=dry_run)

    try:
        try:
            york.unsubscribe(user, maillist)
        except (RequestException, IOError, ValueError), exc:
            logger.exception(u"Проблемы с York")
            raise unsubscribe_york.retry(exc=exc)
        except YorkRepeatableError as exc:
            raise unsubscribe_york.retry(exc=exc, countdown=rnd_delay(20))
    except (RequestException, IOError, ValueError, YorkRepeatableError, YorkError):
        logger.error('Max retries to york exceeded, remove subscription')
        roll_back_task = subscribe_ml.si(context, user, maillist, stype)
        if not autosubscription:
            roll_back_task |= notify_about_metacorp_problem.si(context, user, maillist)

        roll_back_task.delay()


@task(base=MLTask)
def subscribe_cmail(context, user, maillist, type):
    """ Подписка в cmail """
    logger = subscribe_cmail.request.logger
    logger.warning('cmail is obsolete. Skip subscribe_cmail')
    return


@task(base=MLTask, max_retries=42)
def unsubscribe_cmail(context, user, maillist, type):
    """ Отписка в cmail """
    logger = unsubscribe_cmail.request.logger
    logger.warning('cmail is obsolete. Skip unsubscribe_cmail')
    return


@task(base=MLTask)
@transaction.atomic
def create_passport_user(context, data):
    """ Создание пользователя-рассылки в Паспорте """
    logger = create_passport_user.request.logger
    ymail = YmailBackend(logger=logger, dry_run=dry_run)

    if get_suid(data['imap_name']):
        logger.warning(u"Такой пользователь %s уже создан в паспорте",
                       data['imap_name'])
    else:
        try:
            ymail.create_passport_user(data['imap_name'], data['email'])
        except (RequestException, IOError), exc:
            logger.exception(u"Проблемы с сетью при создании пользователя"
                             u" в паспорте")
            raise create_passport_user.retry(exc=exc)

    get_suid(data['imap_name'])  # создаем копию в базе


@task(base=MLTask)
@transaction.atomic
def check_yateam_blackbox_user(context, data):
    """ Проверка на наличие пользователя в yandex-team ЧЯ по входящему email-у"""
    logger = check_yateam_blackbox_user.request.logger

    uid_origin = get_uid_by_email(data['email'])
    uid_imap_name = get_uid_by_email(data['imap_name'])
    if uid_origin or uid_imap_name:
        logger.exception("User with email %s already exists" % data['email'])
        raise UserInYaTeamBlackbox()


@task(base=MLTask)
@transaction.atomic
def check_big_blackbox_user(context, data):
    """ Проверка на наличие пользователя в большом ЧЯ по входящему email-у"""
    logger = check_yateam_blackbox_user.request.logger
    try:
        uid_origin = get_uid_by_email(data['email'], is_yateam=False)
        uid_imap_name = get_uid_by_email(data['imap_name'], is_yateam=False)
    except (RequestException, IOError), exc:
        logger.exception(u"Проблемы с сетью при ")
        raise create_passport_user.retry(exc=exc)

    if uid_origin or uid_imap_name:
        logger.exception("User with email %s already exists" % data['email'])
        raise UserInBigBlackbox()


@task(base=MLTask)
@transaction.atomic
def create_passport_user(context, data):
    """ Создание пользователя-рассылки в Паспорте """
    logger = create_passport_user.request.logger
    ymail = YmailBackend(logger=logger, dry_run=dry_run)

    if get_suid(data['imap_name']):
        logger.warning(u"Такой пользователь %s уже создан в паспорте",
                       data['imap_name'])
    else:
        try:
            ymail.create_passport_user(data['imap_name'], data['email'])
        except (RequestException, IOError), exc:
            logger.exception(u"Проблемы с сетью при создании пользователя"
                             u" в паспорте")
            raise create_passport_user.retry(exc=exc)

    get_suid(data['imap_name'])  # создаем копию в базе


@task(base=MLTask)
@transaction.atomic
def move_user_to_pg(context, data):
    """Переместить пользователя в PG базу"""
    logger = move_user_to_pg.request.logger
    ymail = YmailBackend(logger=logger, dry_run=dry_run)

    suid = get_suid(data['imap_name'])
    try:
        userinfo = get_userinfo(data['imap_name'], by_login=True)
        ymail.change_mdb(suid, userinfo['fields'].get('mdb', settings.YWMI_API_MDB), settings.PG_MIGRATION_DEST_MDB)
    except (RequestException, IOError, PassportException) as exc:
        logger.exception(u"Проблемы при перемещении пользователя в PG")
        raise move_user_to_pg.retry(exc=exc)


@task(base=MLTask, max_retries=42)
def create_maillist_cmail(context, data, user):
    """ Создание рассылки в cmail """
    logger = create_maillist_cmail.request.logger
    logger.warning('cmail is obsolete. Skip create_maillist_cmail')
    return


@task(base=MLTask, max_retries=180)
def check_maillist_creation(context, data):
    """ Проверка наличия рассылки в cmail """
    logger = check_maillist_creation.request.logger
    logger.warning('cmail is obsolete. Skip check_maillist_creation')
    return


@task(base=MLTask, max_retries=42)
def create_maillist_ymail(context, data, parent=None):
    """ Создание рассылки в Почте """
    logger = create_maillist_ymail.request.logger
    ymail = YmailBackend(logger=logger, dry_run=dry_run)

    try:
        ymail.create_maillist(data['imap_name'], parent)
    except (RequestException, IOError, ValueError), exc:
        logger.exception(u"Проблемы с mail.yandex-team.ru")
        raise create_maillist_ymail.retry(exc=exc)
    except YmailError, exc:
        # Timed out while acquiring lock
        if exc.code == 100:
            raise create_maillist_ymail.retry(exc=exc, countdown=rnd_delay(20))
        else:
            raise


@task(base=MLTask, max_retries=42)
def create_maillist_york(context, data):
    """Создание общей папки в York"""
    logger = create_maillist_york.request.logger
    york = YorkBackend(logger=logger, dry_run=dry_run)

    try:
        name = data['name'] if data['domain'] == u'yandex-team.ru' else data['email']
        maillist = MailList.objects.get(name=name)
        fid = york.create(maillist)
        if settings.PG_MIGRATION_STATE == settings.PG_MIGRATION_STATE_PG:
            maillist.fid = fid
            maillist.save()
    except (RequestException, IOError, ValueError), exc:
        logger.exception(u"Проблемы с York")
        raise subscribe_york.retry(exc=exc)
    except YorkRepeatableError as exc:
        raise create_maillist_york.retry(exc=exc, countdown=rnd_delay(20))
    except MailList.DoesNotExist as exc:
        logger.exception(u"Рассылка еще не создана")
        raise create_maillist_york.retry(exc=exc, countdown=rnd_delay(20))


@task(base=MLTask)
@transaction.atomic
def create_maillist_ml(context, data):
    """
    Создание рассылки в ML

    data  =  {'domain': u'yandex-team.ru',
              'email': u'qweqwe123123@yandex-team.ru',
              'imap_name': u'qweqwe123123',
              'info': u'qwe',
              'info_en': u'qwe',
              'is_internal': True,
              'is_open': True,
              'name': u'qweqwe123123',
              'omit_yandex_team': False,
              'responsible': set([u'zubchick@yandex-team.ru'])}
    """
    logger = create_maillist_ml.request.logger
    suid = get_suid(data['imap_name'])
    uid = get_uid(data['imap_name']) or None

    name = data['name'] if data['domain'] == u'yandex-team.ru' else data['email']
    lst, crt = MailList.objects.get_or_create(
        name=name,
        defaults=dict(
            alias=name, email=data['email'],
            letter=data['email'][0].upper(), info=data['info'],
            info_en=data['info_en'], max_length=524288000,  # хз что за цифра
            fsuid=suid, is_imap=True, is_sub=True,
            is_internal=data['is_internal'], is_open=data['is_open'],
            created_dt=datetime.now(),
            before_clean=data.get('before_clean'),
            is_archive=data.get('archive', False),
            use_cmail=False,
            readonly=data.get('readonly', False)))

    if not crt:
        logger.warning(u"%s уже был создан в ML", data['name'])
        return

    for resp in data['responsible']:
        try:
            user = get_user(resp)
        except User.DoesNotExist:
            logger.warning(u"Не могу найти ответственного %s", resp)
            continue
        Owner.objects.get_or_create(list=lst, user=user)

    YandexTeamBackendContext.objects.get_or_create(
        backend_type='yandex_team', maillist=lst,
        defaults=dict(passport_name=data['imap_name'], is_imap=True, is_sub=True,
                      fsuid=suid, uid=uid))


@task(base=MLTask, max_retries=42)
def pdd_create_maillist(context, domain, name, imap_email):
    """ ML-1466 Создание рассылки в Коннекте """
    logger = pdd_create_maillist.request.logger
    directory_operations = directory_api.DirectoryMaillistsOperations()
    try:
        group_uid = directory_operations.create_group_in_organization_by_domain(
            domain_name=domain, group_alias=name,
        )
        assert group_uid  # sanity check
        furita = YFuritaBackend(logger=logger, is_yateam=False)
        rule = {
            'clicker': 'forward',
            "forward_address": imap_email,
            'stop': 0,
            'letter': 'nospam',
            'noconfirm': 1,
            'name': 'fwd: {}'.format(imap_email),
        }
        furita.create(rule, group_uid, 'pg')
    except directory_api.NoOrgWithDomainError:
        logger.error(u'Домен %s не подтвержден в Яндекс.Коннекте' % domain)
        raise pdd_create_maillist.retry(countdown=rnd_delay(20))
    except directory_api.NoAccessToOrganization:
        logger.error(u'yndx.maillists не может создать группу в организации с доменом %s' % domain)
        raise pdd_create_maillist.retry(countdown=rnd_delay(20))
    except directory_api.AliasIsTaken:
        logger.error(u"Уже существует не группа с алиасом %s@%s", name, domain)
        raise RuntimeError('%s@%s is user' % name, domain)
    except directory_api.NoUid:
        logger.error(u'Группа %s@%s уже есть в организации, но у нее нет UID' % name, domain)
        raise RuntimeError('%s@%s group has no UID' % name, domain)
    except (DirectoryAPIError, YFuritaError) as exc:
        logger.exception(u'Не удалось создать группу в организации')
        raise pdd_create_maillist.retry(exc=exc, countdown=rnd_delay(20))


@task(base=MLTask, max_retries=42)
def pdd_register_domain(context, domain):
    """ ML-1466 Создание домена в ПДД """
    logger = pdd_register_domain.request.logger

    pdd = PDDBackend(logger=logger, dry_run=dry_run)
    try:
        pdd.register_domain_for_corp_maillists(domain=domain)
    except (RequestException, IOError), exc:
        logger.warning(u"Проблемы доступностью ПДД: %s", exc)
        raise pdd_register_domain.retry(exc=exc)
    except PDDError as exc:
        if exc.code == 'not_allowed':
            logger.error(u"Ошибка %s при вызове pdd.register_domain_for_corp_maillists, домен в ПДД уже существует, и у него другой админ" % exc.code)
            raise
        elif exc.code:
            logger.warning(u"Ошибка %s при вызове pdd.register_domain_for_corp_maillists, повторяем" % exc.code)
            raise pdd_register_domain.retry(exc=exc, countdown=rnd_delay(20))
        else:
            raise


@task(base=MLTask, max_retries=42)
def passport_set_altdomain(context, imap_name, email):
    """ ML-1463 Создание алиаса в паспорте для некоторых типов доменов. """
    logger = passport_set_altdomain.request.logger
    suid = get_suid(imap_name)
    ymail = YmailBackend(logger=logger, dry_run=dry_run)
    try:
        ymail.passport_alias_altdomain(suid=suid, email=email)
    except (RequestException, IOError), exc:
        logger.warning(u"Проблемы доступностью паспорта: %s", exc)
        raise passport_set_altdomain.retry(exc=exc, countdown=rnd_delay(10))
    except YmailError, exc:
        if exc.code >= 500:
            logger.warning(u"Ошибка %s при вызове ymail.passport_alias_altdomain, повторяем" % exc.code)
            raise passport_set_altdomain.retry(exc=exc, countdown=rnd_delay(20))
        else:
            raise


@task(base=MLTask, max_retries=42)
def change_list_properties_cmail(context, maillist, internal=None, open=None):
    """ Изменение свойств рассылки в cmail """
    logger = change_list_properties_cmail.request.logger
    logger.warning('cmail is obsolete. Skip change_list_properties_cmail')
    return


@task(base=MLTask, max_retries=42)
def move_list_ymail(context, maillist, parent_maillist):
    """ Перенос фолдеров по дереву в ymail. """
    logger = move_list_ymail.request.logger
    maillist = get_list(maillist)
    parent_maillist = get_list(parent_maillist) if parent_maillist else None
    ymail = YmailBackend(logger=logger, dry_run=dry_run)

    try:

        ymail.set_parent(maillist, parent_maillist)
    except (RequestException, IOError, ValueError), exc:
        logger.exception(u"Ошибка при вызове ymail.set_parent, повторяем")
        raise move_list_ymail.retry(exc=exc)
    except YmailError, exc:
        if exc.code >= 500:
            logger.warning(u"Ошибка %s при вызове ymail.set_parent, повторяем" % exc.code)
            raise move_list_ymail.retry(exc=exc, countdown=rnd_delay(20))
        else:
            logger.exception(u"Не можем сменить родителя: ошибка %s при вызове ymail.set_parent", exc.code)
            raise


@task(base=MLTask, max_retries=42)
def set_shared_flags_ymail(context, maillist, state):
    """ Включаем общие флажки """
    logger = set_shared_flags_ymail.request.logger
    maillist = get_list(maillist)
    ymail = YmailBackend(logger=logger, dry_run=dry_run)

    try:
        if state is not None:
            res = ymail.set_shared_flags(maillist, state)
            logger.info("Result from set_shared_flags: %s", res)
    except (RequestException, IOError, ValueError), exc:
        logger.exception(u"Проблемы с ymail(shared_flags)")
        raise set_shared_flags_ymail.retry(exc=exc)
    except YmailError, exc:
        # 30024 - значение свойства рассылки равно изменяемому
        if exc.code >= 500:
            logger.error(u"Error %d.", exc.code)
            raise set_shared_flags_ymail.retry(exc=exc, countdown=rnd_delay(20))
        else:
            raise
    except Exception as ex:
        logger.error(u"Unknown error: %s.", ex)
        raise set_shared_flags_ymail.retry(exc=ex, countdown=rnd_delay(20))


@task(base=MLTask, max_retries=42)
def change_ymail_archiving_options(context, maillist, is_archive=True, before_clean=None, max_size=None,
                                   archiving_type=ARCHIVING_TYPE.CL):
    """ Удаление писем через before_clean дней, max_size писем, которое нужно оставлять в папке"""
    maillist = get_list(maillist)
    logger = change_ymail_archiving_options.request.logger

    york = YorkBackend(logger=logger, dry_run=dry_run)
    try:
        try:
            if is_archive:
                york.set_archivation_rule(maillist, archiving_type, keep_days=before_clean, max_size=max_size)
            else:
                york.remove_archivation_rule(maillist)
        except (RequestException, IOError, ValueError), exc:
            logger.exception(u"Проблемы с York")
            raise change_ymail_archiving_options.retry(exc=exc)
        except YorkRepeatableError as exc:
            raise change_ymail_archiving_options.retry(exc=exc, countdown=rnd_delay(20))
    except (RequestException, IOError, ValueError, YorkRepeatableError, YorkError):
        logger.error('Max retries to york exceeded, could not change archiving options')
        raise


@task(base=MLTask)
@transaction.atomic
def create_permission(context, user, maillist, approved=True):
    """ Создание доступа """
    user = get_user(user)
    maillist = get_list(maillist)

    if not maillist.is_open:
        if approved:
            ListPermission.objects.grant_permission(maillist, user)
        else:
            ListPermission.objects.get_or_create(list=maillist, user=user,
                                                 type=Type.obects.get(name='read'),
                                                 defaults={'approved': False})


@task(base=MLTask)
@transaction.atomic
def delete_permission(context, user, maillist):
    """ Отзыв доступа """
    user = get_user(user)
    maillist = get_list(maillist)
    ListPermission.objects.filter(user=user, list=maillist).delete()


@task(base=MLTask)
def sync_corp_domains(context):
    """
    Синхронизируем корпоративные домены из Cmail в ML
    """
    logger = sync_corp_domains.request.logger
    logger.warning('cmail is obsolete. Skip sync_corp_domains')
    return


@task(base=MLTask)
def create_domain(context, domain):
    """
    Добавляем новые домены из ML в Cmail
    """
    logger = create_domain.request.logger
    logger.warning('cmail is obsolete. Skip create_domain')
    return


@task(base=MLTask)
def create_separate_folder_and_filter(context, user, maillist):
    maillist = get_list(maillist)
    logger = create_separate_folder_and_filter.request.logger

    if isinstance(user, basestring) and '@' in user:
        try:
            user = get_user(user)
        except User.DoesNotExist:
            logger.info(u"Не создаем для внеших пользователей", user)
            return
    else:
        user = get_user(user)

    mdb = get_userinfo(user.staff.uid, by_login=False).get('fields', {}).get('mdb', settings.YWMI_API_MDB)
    ywmi = YWMIBackend(logger=logger, dry_run=dry_run, mdb=mdb)
    special_folder = SpecialMailListFolder(dry_run=dry_run, mdb=mdb)
    mops = MopsBackend(logger=logger, dry_run=dry_run, mdb=mdb)

    try:
        folders = ywmi.folders(user)

        maillist_folder_id = None
        for fid, folder in folders.iteritems():
            if folder['name'] == maillist.name:
                maillist_folder_id = fid
                break

        if not maillist_folder_id:
            maillist_folder_id = mops.create_folder(maillist.name, user)

        if not special_folder.find_rule(maillist, user, maillist_folder_id):
            special_folder.create_rule(maillist, user, maillist_folder_id)

    except (RequestException, IOError, ValueError), exc:
        logger.exception(u"Проблемы с mail.yandex-team.ru")
        raise create_separate_folder_and_filter.retry(exc=exc)
    except (MopsError, YWMIError, YFuritaError) as exc:
        # Timed out while acquiring lock
        if exc.code == 100:
            raise create_separate_folder_and_filter.retry(exc=exc, countdown=rnd_delay(20))
        else:
            raise


@task(base=MLTask)
def notify_about_metacorp_problem(context, user, maillist, unsubscribe=False):
    try:
        user = get_user(user)
        maillist = get_list(maillist)
        notification = YaMailSubscriptionProblem(context={'list': maillist, 'unsubscribe': unsubscribe})
        notification.send([user.email], cc=['ml-dev@yandex-team.ru'])
    except (User.DoesNotExist, MailList.DoesNotExist):
        notify_about_metacorp_problem.request.logger.warning('Metacorp notifications wrong user or mailllist')


@task(base=MLTask)
def notify_about_duplicate_folder_problem(context, user, maillist, autosubscription):
    try:
        user = get_user(user)
        maillist = get_list(maillist)
        notification = YorkDuplicateFolderProblem(context={'list': maillist, 'autosubscription': autosubscription})
        notification.send([user.email])
    except (User.DoesNotExist, MailList.DoesNotExist):
        notify_about_metacorp_problem.request.logger.warning('Metacorp notifications wrong user or mailllist')
