# coding: utf-8

import logging
import socket
import yenv
from blackbox import Blackbox

logger = logging.getLogger(__name__)


def get_ip_address():
    fqdn = socket.getfqdn()
    try:
        ip = socket.getaddrinfo(fqdn, None, 0, 0, socket.SOL_TCP)[0][4][0]
    except IndexError:
        ip = socket.getaddrinfo(fqdn, None, 0, 0, socket.SOL_TCP)[1][4][0]

    return ip


SERVICE_IP = get_ip_address()


class UserInYaTeamBlackbox(Exception):
    """
    Исключение про существование пользователя во внутреннем yandex-team ЧЯ
    """
    pass


class UserInBigBlackbox(Exception):
    """
    Исключение про существование пользователя в Большом ЧЯ
    """
    pass


def get_blackbox(is_yateam=True):
    return Blackbox(url=Blackbox.URLS['intranet' if is_yateam else 'other'][yenv.type],
                    timeout=(0.45, 1.0))


def get_userinfo_by_login(passport_name, sid=2, is_yateam=True):
    bb = get_blackbox(is_yateam)
    return bb.userinfo(
        passport_name,
        userip=SERVICE_IP,
        dbfields={
            'subscription.suid.%d' % sid: 'suid'
        },
        by_login=True
    )


def get_userinfo(uid_or_email, by_login=True, sid='smtp', is_yateam=True):
    bb = get_blackbox(is_yateam)
    return bb.userinfo(
        uid_or_email,
        userip=SERVICE_IP,
        dbfields={
            'accounts.login.uid': 'uid',
            'subscription.suid.2': 'suid',
            'hosts.db_id.2': 'mdb'
        },
        by_login=by_login,
        sid=sid,
        aliases='9',
        emails='getall',
    )


def get_uid_by_email(email, is_yateam=True):
    response = get_userinfo(email, is_yateam=is_yateam)
    return response.get('uid', 0)


def get_all_emails_by_uid(uid, is_yateam=True):
    response = get_userinfo(uid, by_login=False, is_yateam=is_yateam)
    emails = [user.get("address") for user in response.get('emails', [])]
    return emails


def get_alias_by_uid(uid, is_yateam=True):
    response = get_userinfo(uid, by_login=False, is_yateam=is_yateam)
    return response['fields'].get('aliases')


def get_suid(passport_name, is_yateam=True):
    response = get_userinfo_by_login(passport_name, is_yateam=is_yateam)
    suid = response.get('fields', {}).get('suid')
    if suid is None:
        error = response.get('error')
        if error is not None:
            logger.warning(u"Blackbox error while requesting suid: %s", error)
    return suid


def get_uid(passport_name, is_yateam=True):
    response = get_userinfo_by_login(passport_name, is_yateam=is_yateam)
    return response.get('uid')
