# -*- coding: utf-8 -*-

from urllib import urlencode

from django.db import models
from django.core.cache import cache


def generate_key_from_dict(mapping):
    return urlencode(sorted(mapping.items()))


class GetCachingManager(models.Manager):
    '''
    Simple caching manager that caches object returned by QuerySet.get()
    method.

    Overrides get() thus hitting db only once in specified timeout for each
    lookup parameters.
    '''

    def __init__(self, timeout=None):
        self._cache_timeout = timeout
        super(GetCachingManager, self).__init__()

    def contribute_to_class(self, model, name):
        super(GetCachingManager, self).contribute_to_class(model, name)
        self._cache_key_prefix = ':'.join([
            self.model._meta.app_label,
            self.model.__name__])

    def _generate_get_key(self, lookup):
        return ':'.join(
            (self._cache_key_prefix, generate_key_from_dict(lookup)))

    def get(self, **lookup):
        cache_key = self._generate_get_key(lookup)
        cached_result = cache.get(cache_key)
        if cached_result is not None:
            return cached_result
        else:
            # raises as always on error
            result = super(GetCachingManager, self).get(**lookup)
            cache.set(cache_key, result, self._cache_timeout)
            return result
