# encoding: utf-8
from __future__ import absolute_import

import dns.resolver
from django.conf import settings


def _resolve_cname(domain):
    """
    Раскрываем алиас (CNAME) домена.
    """
    try:
        for r in dns.resolver.query(domain, 'CNAME'):
            if r.target:
                return _resolve_cname(r.target.to_text())
    except dns.resolver.NoAnswer:
        pass
    return domain


def mx(domain):
    """
    Возвращаем MX-записи домена
    :param domain:
    :return: ['mail1.yandex.ru', 'mail2.yandex.ru']

    >>> mx('passport.yandex.ru')
    ['mx-corp.yandex.ru.']

    """
    try:
        domain = _resolve_cname(domain)
        return [r.exchange.to_text() for r in dns.resolver.query(domain, 'MX')]
    except dns.resolver.NoAnswer:
        return []
    except dns.resolver.NXDOMAIN:
        raise DomainNotExists()


def exists(domain):
    """
    Проверяем существует ли домен:

    >>> exists('bla-bla.yandex-team.ru')
    False
    """
    try:
        dns.resolver.query(domain, 'A')
        return True
    except dns.resolver.NXDOMAIN:
        return False
    except dns.resolver.NoAnswer:
        return bool(mx(domain))


class DomainError(Exception):
    description = ''


class DomainNotExists(DomainError):
    description = 'Domain does not exists'


class DomainHasWrongMx(DomainError):
    description = 'Domain has wrong MX records'


class Domain(object):
    def __init__(self, name):
        name = name.lower()
        self._name = name

        self._mxs = None
        self._exists = None

    def validate(self):
        self._mxs = set(mx(self.name))
        self._exists = exists(self._name)

        if not self.exists:
            raise DomainNotExists

        mx_ok = self.is_pdd_mx or self.is_corp_mx or self.has_no_mx
        if not mx_ok:
            raise DomainHasWrongMx

    @property
    def mxs(self):
        return self._mxs

    @property
    def is_pdd_mx(self):
        # пустые MX тоже считаем за внешние
        pdd_mxs = set(settings.MX_PDD)
        return self._mxs.issubset(pdd_mxs)

    @property
    def is_corp_mx(self):
        corp_mxs = set(settings.MX_CORP)
        return self._mxs.issubset(corp_mxs) and bool(self._mxs)

    @property
    def has_no_mx(self):
        return not self._mxs

    @property
    def is_altdomain(self):
        return self.is_corp_mx and (self._name in settings.PASSPORT_ALIASES)

    @property
    def exists(self):
        return self._exists

    @property
    def name(self):
        return self._name
