# -*- coding: utf-8 -*-
import re
from abc import ABCMeta, abstractmethod

from django import forms
from django.conf import settings
from django.db.models import Q
from django.utils.translation import ugettext

from django_intranet_stuff.models import Staff

from mlcore.ml.models import MailList, CreateListLock

# Ограничение на длину логина в pdd
MAX_PDD_LENGTH = 40
# Ограничение на длину логина в yandex-team.ru
MAX_YANDEX_TEAM_LENGTH = 100
# Максимальная длина pdd-домена, чтобы в imap_name-виде(login-at-domain) не привысить MAX_YANDEX_TEAM_LENGTH
MAX_PDD_DOMAIN_LENGTH = MAX_YANDEX_TEAM_LENGTH - len('-at-') - 1

PROHIBITED_PDD_SYMBOLS_RE = r'[^a-zA-Z0-9._-]'
PROHIBITED_YANDEX_TEAM_SYMBOLS_RE = r'[^a-zA-Z0-9_-]'

ALLOWED_PDD_SYMBOLS = '[a-z,A-Z,0-9,.,_,-]'
ALLOWED_YANDEX_TEAM_SYMBOLS = '[a-z,A-Z,0-9,_,-]'


class EmailValidator:
    __metaclass__ = ABCMeta

    @abstractmethod
    def validate(self, email):
        pass


class LoginValidator(EmailValidator):
    max_length = MAX_PDD_LENGTH
    prohibited_symbols_re = re.compile(PROHIBITED_PDD_SYMBOLS_RE)
    allowed_symbols = ALLOWED_PDD_SYMBOLS

    def __init__(self):
        pass

    def validate(self, email):
        login = email.split('@', 1)[0]
        errors = []

        if len(login) > self.max_length:
            errors.append(ugettext(u"ERROR.LOGIN_TOO_LONG %(max_length)s") % {'max_length': self.max_length})

        if self.prohibited_symbols_re.search(login):
            errors.append(ugettext(u"ERROR.LOGIN_CONTAINS_PROHIBITED_SYMBOLS %(symbols)s")
                          % {'symbols': self.allowed_symbols})

        return errors


class PddLoginValidator(LoginValidator):
    pass


class YandexTeamLoginValidator(LoginValidator):
    """Для логинов в домене yandex-team применяются другие правила"""
    max_length = MAX_YANDEX_TEAM_LENGTH
    prohibited_symbols_re = re.compile(PROHIBITED_YANDEX_TEAM_SYMBOLS_RE)
    allowed_symbols = ALLOWED_YANDEX_TEAM_SYMBOLS

    def validate(self, email):
        errors = []
        errors.extend(super(YandexTeamLoginValidator, self).validate(email))
        login = email.split('@', 1)[0]

        # не может начинаться на дефис
        if login[0] == '-':
            errors.append(ugettext(u"ERROR.LOGIN_STARTS_WITH_HYPHEN"))

        # не может заканчиваться на дефис
        if login[-1] == '-':
            errors.append(ugettext(u"ERROR.LOGIN_ENDS_WITH_HYPHEN"))

        return errors


class MaillistEmailValidator(EmailValidator):
    def __init__(self):
        pass

    @staticmethod
    def _to_imap_name_and_domain(email):
        """
        Преобразуем email в imap_name-вид и imap_domain.
        Если email вида yandex-team.com, yandex-team.com.tr, support.yandex.ru, и др, -> name, domain
        Если приходит email с доменом yandex-team.ru, -> name, ''
        Ex.1: api-direct-support@support.yandex-team.com.tr -> api-direct-support-at-support-yandex-team-com-tr,
              support-yandex-team-com-tr
        Ex.2: api-direct-support@yandex-team.ru -> api-direct-support, ''
        """
        name, domain = email.split('@', 1)
        if domain == 'yandex-team.ru':
            return name, ''

        domain_suffix = domain.replace('.', '-')
        imap_name = u'-at-'.join([name.replace('.', '-'), domain_suffix])
        return imap_name, domain_suffix

    def validate(self, email):
        errors = []
        # проверяем есть ли уже рассылки/пользователи в стаффе с таким же именем
        name, domain = email.split('@', 1)
        if (MailList.objects.filter(email=email).exists() or
                CreateListLock.objects.filter(name=email).exists()):
            errors.append(ugettext(u"ERROR.LIST_ALREADY_EXISTS %(list)s") % {'list': email})
        elif (domain in settings.YANDEX_TEAM_DOMAINS
              and Staff.objects.filter(Q(login=name) | Q(work_email=email)).exists()):
            errors.append(ugettext(u"User with such name already exists"))
        if errors:
            return errors
        # если pdd-домен, то проверяем, удовлетворяет ли имя правилам для pdd
        imap_name, imap_domain = self._to_imap_name_and_domain(email)
        if imap_domain:
            errors.extend(PddLoginValidator().validate(email))
            if len(imap_name) > MAX_YANDEX_TEAM_LENGTH:
                len_imap_domain = len(imap_domain) + len('-at-')
                limit_name = MAX_YANDEX_TEAM_LENGTH - len_imap_domain

                if limit_name <= 0:
                    errors.append(ugettext(u"ERROR.IMAP_DOMAIN_TOO_LONG %(N)s") % {'N': MAX_PDD_DOMAIN_LENGTH})
                else:
                    errors.append(ugettext(u"ERROR.IMAP_LOGIN_TOO_LONG %(N)s") % {'N': limit_name})
            if errors:
                return errors

        # проверяем, что имя валидно в домене yandex-team
        errors.extend(YandexTeamLoginValidator().validate(imap_name))

        return errors

    def __call__(self, value):
        errors = self.validate(email=value)
        if errors:
            raise forms.ValidationError(';'.join(errors))
