# coding: utf-8
'''
ORM-wrappers around mdb30x models (that lays in oracle)
'''
from itertools import imap
# from operator import attrgetter
from datetime import datetime

from requests.exceptions import RequestException

from django.conf import settings

from mlcore.utils.basic import memoizible_property
from mlcore.utils.getters import get_list_uid
from mlcore.interaction.ywmi import YWMIBackend, YWMIError

dry_run = settings.DRY_RUN

class ThreadList(object):

    def __init__(self, thread_list, page_number, has_next, results_on_page=25):

        self.objects_on_page = results_on_page
        self.page_number = page_number
        self._has_next = has_next
        self._list = thread_list

    def __len__(self):
        return len(self._list) if self._list else 0

    def __iter__(self):
        if self._list is not None:
            return iter(self._list)
        else:
            return iter([])

    def get_from_emails(self):
        for t in self._list:
            yield getattr(t, 'from')

    def has_previous(self):
        return bool(self.page_number)

    def has_next(self):
        return self._has_next

    @property
    def previous_page_number(self):
        return self.page_number - 1

    @property
    def next_page_number(self):
        return self.page_number + 1

    @property
    def previous_list(self):
        return range(0, self.page_number)

    def list(self):
        return self._list


class ArchiveProxy(object):
    """
    Чтобы не переделывать весь старый код проксируем данные из апи так
    будто это выборка из базы orm'ом.

    """

    def __init__(self, obj):
        super(ArchiveProxy, self).__init__()
        self.raw = obj
        self._map = {
            'from': (self.convert_from, 'from'),
            'received_date': (self.convert_date, 'date'),
        }

    def __getattr__(self, name):
        if name in self._map:
            res = self._map[name]

            if isinstance(res, tuple):
                func, nm = res
                return func(self.raw[nm])
            else:
                return self.raw[res]

        if name in self.raw:
            return self.raw[name]

        raise AttributeError("'Thread' object has no attribute '%s'" % name)

    def convert_date(self, timestamp):
        return datetime.fromtimestamp(timestamp)

    def build_email(self, values):
        try:
            return [u'{local}@{domain}'.format(**x) for x in values]
        except KeyError:
            return []

    def convert_from(self, from_list):
        """ вычленяет емейл """
        emails = self.build_email(from_list)
        return emails[0] if emails else ''


class ThreadProxy(ArchiveProxy):
    def __init__(self, obj):
        super(ThreadProxy, self).__init__(obj)
        self._map.update({
            'thread_id': 'threadId',
            'msg_cnt': 'threadCount',
        })    # маппинг имен в старые атрибуты


class MessageProxy(ArchiveProxy):
    """ Работает так же как тред прокси """

    def __init__(self, obj):
        super(MessageProxy, self).__init__(obj)
        self._map.update({
            'to': (self.build_email, 'to'),
            'cc': (self.build_email, 'cc'),
            'reply_to': 'replyTo',
        })

    def set_duplicates(self, duplicate_list):
        self._duplicates = duplicate_list
        self._duplicate_by_fid = dict(
            (int(d.fid), int(d.mid)) for d in duplicate_list)

    @memoizible_property
    def fid_list(self):
        result = []
        for m in self._duplicates:
            try:
                result.append(int(m.fid))
            except ValueError:
                continue
        return result

    def map_fids(self, mapping):
        return filter(None, imap(mapping.get, self.fid_list))

    def filter_first_mid(self, fids):
        for fid in fids:
            mid = self._duplicate_by_fid.get(fid)
            if mid:
                return mid
        return None

    @property
    def all_recipients(self):
        return self.to + self.cc

    @property
    def all_mentioned(self):
        return self.to + self.cc + [getattr(self, 'from')]

    def map_recipients(self, mapping):
        return filter(None, imap(mapping.get, self.build_email(self.reply_to)))


class ThreadReel(object):

    def __init__(self, messages_info):
        self._messages = []
        # WMI не возвращает данных для дедубликации писем
        # for msg_id_hash, subiter in groupby(messages_info,
        #                                     attrgetter('msg_id_hash')):
        for message in messages_info:
            message.set_duplicates((message,))
            self._messages.append(message)

    def __len__(self):
        return len(self._messages)

    @property
    def thread_subject(self):
        return self._messages[0].subject if self._messages else None

    @memoizible_property
    def fid_list(self):
        for msg_in_reel in self._messages:
            for fid in msg_in_reel.fid_list:
                yield fid

    @property
    def messages(self):
        return self._messages

    def get_mid_list(self):
        for msg_in_reel in self._messages:
            yield msg_in_reel.mid

    def __iter__(self):
        return iter(self.full_messages)


def get_threads(maillist, page_number, results_on_page=25):
    fuid = get_list_uid(maillist)
    client = YWMIBackend(timeout=7, dry_run=dry_run)
    try:
        res = client.threads_list(fuid, maillist, page_number, results_on_page, select_one_next=True)
    except RequestException:
        return
    except YWMIError:
        return ThreadList([], 0, False)

    if not res:
        return ThreadList([], 0, False)

    has_next = False
    if len(res) > results_on_page:
        has_next = True
        res.pop()

    return ThreadList(map(ThreadProxy, res),
                      page_number, has_next, results_on_page)


def get_thread(maillist, thread_id):
    fuid = get_list_uid(maillist)
    client = YWMIBackend(timeout=3)
    try:
        res = client.mails_in_thread(fuid, thread_id)
    except RequestException:
        return
    except YWMIError:
        return ThreadReel([])

    return ThreadReel(map(MessageProxy, res))
