# encoding: utf-8

import functools
from lxml import etree as ET
from email.mime.base import MIMEBase

def memoize(obj):
    cache = obj.cache = {}
    @functools.wraps(obj)
    def memoizer(*args, **kwargs):
        key = str(args) + str(kwargs)
        if key not in cache:
            cache[key] = obj(*args, **kwargs)
        return cache[key]
    return memoizer


class Message(object):

    """
    Объект-хэлпер для доступа к сообщению, полученному из мулькагейта
    """

    def __init__(self, mulcagate, st_id, mid=None):
        self.mulcagate = mulcagate
        self.st_id = st_id
        self.mid = mid

    def get(self, **kw):
        return self.mulcagate.get(st_id=self.st_id, **kw)

    @memoize
    def get_part(self, id):
        # вернём part с номером id
        return self.parts[id]

    def __getitem__(self, id):
        return self.get_part(id)

    #@memoize
    #def _get_part_meta(self, id):
    #    pass

    @property
    @memoize
    def parts(self):
        return MessageParts(message=self)

    @memoize
    def get_parts(self):
        # get_parts для совместимости с mail.yandex.ru/api и ymailapi возвращает только текстовый контент письма
        return filter(lambda p: p.is_message_body, self.parts )

    @property
    @memoize
    def headers(self):
        return MulcaHeaders.from_message(message=self)

    @memoize
    def get_attachments(self):
        # get_attachments возвращает аттачменты
        return filter(lambda p: p.is_attachment, self.parts )
    
    @memoize
    def get_mentioned_emails(self):
        # все упоминаемые в заголовках email-ы (from, to, cc, bcc)
        # TODO: реализовать
        return []
        



class MulcaHeaders(object):

    """
    Хэлпер для работы с хэдерами мулькагейта.

    Хэдеры - они же мета-данные - это xml с информацией о частях письма.

    Например:

    <message>
    <part id="1" offset="6740" length="16375"
        content_type.type="multipart"
        content_type.subtype="related"
        content_type.charset="US-ASCII"
        content_transfer_encoding="7bit">
        <part id="1.1" offset="6920" length="7747"
            content_type.type="multipart"
            content_type.subtype="alternative"
            content_type.charset="US-ASCII"
            content_transfer_encoding="7bit">
            </part>
            <part id="1.1.2" offset="9308" length="5290"
                content_type.type="text"
                content_type.subtype="html"
                content_type.charset="koi8-r"
                content_transfer_encoding="quoted-printable">
            </part>
        </part>
        <part id="1.2" offset="15066" length="7978"
            content_type.type="image"
            content_type.subtype="jpeg"
            content_type.charset="US-ASCII"
            content_type.name="image003.jpg"
            content_transfer_encoding="base64"
            content_disposition.value="inline"
            content_disposition.filename="image003.jpg"
            content_id="image003.jpg@01CEEF5C.AF2C0350">
        </part>
    </part>
    </message>

    """

    def __init__(self, message):
        self.message = message

    @classmethod
    def from_message(cls, message):
        return cls(message=message)

    @property
    @memoize
    def content(self):
        return self.message.get(gettype='xml').content

    @property
    @memoize
    def tree(self):
        return ET.XML(self.content)

    def __str__(self):
        return self.content

    def __getitem__(self, id):
        return self.tree.xpath("//part[@id='%s']" % id)[0]


class MulcaPartHeaders(object):

    """
    Холдер для мулечных хэдеров одного part-а

    Пример:

    <part id="1.2" offset="15066" length="7978"
        content_type.type="image"
        content_type.subtype="jpeg"
        content_type.charset="US-ASCII"
        content_type.name="image003.jpg"
        content_transfer_encoding="base64"
        content_disposition.value="inline"
        content_disposition.filename="image003.jpg"
        content_id="image003.jpg@01CEEF5C.AF2C0350">
    </part>

    """

    def __init__(self, message, xml):
        self.xml = xml
        self.message = message

    def get(self, k, default=None):
        return self.xml.get(k) or default

    def __getitem__(self, k):
        return self.get(k)

    @property
    def id(self):
        return self.xml.get("id")

    @property
    def content_id(self):
        return self.xml.get("content_id")
    
    @property
    def content_type(self):
        return self.xml.get("content_type.type")

    @property
    def content_subtype(self):
        return self.xml.get("content_type.subtype")

    @property
    def charset(self):
        return self.xml.get("content_type.charset")

    @property
    def content_transfer_encoding(self):
        return self.xml.get("content_transfer_encoding")

    @property
    def content_disposition_filename(self):
        return self.xml.get("content_disposition.filename")

    def as_dict(self):
        MAPPING =  (  ('id', 'id'),
                      ('content_type', 'content_type.type'),
                      ('content_subtype', 'content_type.subtype'),
                      ('charset', 'content_type.charset'),
                      ('content_transfer_encoding', 'content_transfer_encoding')
         )

        d = [ (id, self.get(mapped_id)) for (id, mapped_id) in MAPPING ]
        return dict( d )


class MessageParts(object):

    """
    Прокси-объект для доступа к частям письма.
    """

    def __init__(self, message):
        self.message = message
        self._cache = {}

    def __getitem__(self, k):
        r = self._cache.get(k)
        if r is None:
            r = MessagePart.from_headers(MulcaPartHeaders(message=self.message,
                                                          xml=self.message.headers[k]))
            self._cache[r.id] = r
        return r

    def __iter__(self):
        for el in self.message.headers.tree.xpath("//part"):
            id = el.get('id')
            part = self._cache.get(id)
            if part is None:
                part = MessagePart.from_headers(MulcaPartHeaders(message=self.message, xml=el))
                self._cache[part.id] = part
            yield part



class MessagePart(object):

    """ Объект представляет одну часть письма """

    def __init__(self, headers=None):
        self.headers = headers
        self.message = self.headers.message
        self.id = headers.id # это part id
        self.content_id = headers.content_id

    @classmethod
    def from_headers(cls, headers):
        return cls(headers=headers)


    def _get(self):
        """ Загружает контент из мулькагейта """
        return self.message.get(gettype='part', part=self.id)

    @property
    def subtype(self):
        return self.headers.content_subtype

    @property
    @memoize
    def _mime_message(self):
        """ """
        msg = MIMEBase(self.headers.content_type,
                       self.headers.content_subtype,
                       charset=self.headers.charset)
        msg.add_header('content-transfer-encoding', self.headers.content_transfer_encoding)
        msg.set_payload(self._get().content)
        return msg


    @property
    @memoize
    def text(self):
        c = self._mime_message.get_payload(decode=True)
        return unicode(c, self.headers.charset)


    @property
    @memoize
    def content(self):
        c = self._mime_message.get_payload(decode=True)
        return c


    @property
    def is_message_body(self):
        return self.headers.content_type in ('text', )

    @property
    def is_attachment(self):
        return self.headers.get('content_disposition.value') in ('attachment', )

    @property
    def disposition_filename(self):
        return self.headers.get('content_disposition.filename')

    @property
    def hid(self):
        return self.id

    @property
    def length(self):
        try:
            return int(self.headers.get('length'))
        except ValueError:
            return 0



