# encoding: utf-8
import os

import requests
import logging
import time
from django.conf import settings

from mlcore.utils.tvm2 import get_tvm_2_header
from ya_directory.exceptions import (
    DirectoryAPIError, DirectoryRepeatResponse, DirectoryNotFound, DirectoryForbidden,
    Unprocessable, DirectoryResponseError, DirectoryAlreadyExistsError,
)

logger = logging.getLogger(__name__)


class DirectoryAPI(object):
    url = settings.DIRECTORY_API_URL
    timeout = settings.DIRECTORY_TIMEOUT
    retry_timeout = settings.DIRECTORY_RETRY_TIMEOUT

    USER_ALL_FIELDS = (
        'about', 'aliases', 'avatar_id', 'birthday', 'contacts', 'created', 'department', 'department_id',
        'departments', 'email', 'external_id', 'first_name', 'gender', 'groups', 'id', 'is_admin',
        'is_dismissed', 'is_enabled', 'is_outstaff', 'is_robot', 'karma', 'language', 'last_name', 'login',
        'middle_name', 'name', 'nickname', 'org_id', 'position', 'position_plain', 'recovery_email', 'role',
        'service_slug', 'services', 'timezone', 'updated_at', 'user_type')

    def __init__(self, token=settings.DIRECTORY_API_TOKEN, admin_uid=None):
        self.token = token
        self.admin_uid = admin_uid
        self.ip = os.environ.get('QLOUD_IPV6', '127.0.0.1').split('/')[0]

    def _check_response_code(self, response):
        if 200 <= response.status_code < 300:
            return
        if response.status_code == 404:
            raise DirectoryNotFound(message='resource not found', response=response)
        if response.status_code == 403:
            raise DirectoryForbidden(message='no organization for user', response=response)
        if response.status_code >= 500:
            raise DirectoryRepeatResponse(response=response)
        if response.status_code == 409:
            raise DirectoryAlreadyExistsError(message='object already exists', response=response)
        if response.status_code >= 422:
            raise Unprocessable(message=response.text, response=response)
        if response.status_code >= 400:
            code = {
                '401': 'unauthorized request',
                '405': 'method not allowed'
            }
            if str(response.status_code) in code:
                raise DirectoryResponseError(message=code[str(response.status_code)], response=response)
            else:
                raise DirectoryAPIError(message='unhandled return code', response=response)
        else:
            raise DirectoryAPIError(message='unhandled return code', response=response)

    def _endpoint_url(self, method):
        return '{0}{1}'.format(self.url, method)

    def _make_request(self, url, headers, method='get', params=None, data=None):
        ts = time.time()
        logger.info(u"Вызов коннекта [%s] %s headers=%s params=%s, data=%s",
                    method, url, headers, params, data)
        headers.update(get_tvm_2_header(settings.DIRECTORY_TVM_ID))

        for i, retry_timeout in enumerate(self.retry_timeout):
            try:
                try:
                    request_method = getattr(requests, method)
                    r = request_method(
                        url,
                        headers=headers,
                        timeout=self.timeout,
                        params=params,
                        json=data,
                        verify=False)
                    logger.info(u"Коннект вернул: [%s %s] %s, через %s сек", method, r.status_code,
                                r.text, time.time() - ts)
                    self._check_response_code(r)

                    return r.json()
                except requests.ConnectionError:
                    logger.exception('directory connection error')
                    raise DirectoryRepeatResponse
            except DirectoryRepeatResponse:
                request_time = time.time() - ts
                remaining_timeout = self.timeout - request_time
                if remaining_timeout < 0:
                    remaining_timeout = 0
                time.sleep(remaining_timeout + self.retry_timeout[i])

                if i == len(self.retry_timeout):
                    raise DirectoryAPIError('retry timeout exceeded')

    def get_group_info(self, org_id, group_id, fields=('id', 'email', 'members', 'member_of', 'uid')):
        headers = {
            'X-USER-IP': self.ip,
            'X-ORG-ID': str(org_id),
        }

        if self.token:
            headers['Authorization'] = 'token {}'.format(self.token)

        if self.admin_uid:
            headers['X-UID'] = str(self.admin_uid)

        return self._make_request(
            self._endpoint_url('groups/{0}/'.format(group_id)),
            params={'fields': ','.join(fields)},
            headers=headers
        )

    def get_domains(self, domain, fields=None):
        if fields:
            fields = ','.join(fields)

        return self._make_request(
            self._endpoint_url('domains/'),
            headers={
                'X-USER-IP': self.ip,
                'X-UID': str(self.admin_uid)
            },
            params={
                'name': domain,
                'fields': fields
            }
        )

    def create_org(self, domain_name):
        """Создать организацию с неподтвержденным доменом domain_name и админом self.admin_uid"""
        return self._make_request(
            self._endpoint_url('organization/with-domain/'),
            method='post',
            headers={
                'X-USER-IP': self.ip,
                'X-UID': str(self.admin_uid)
            },
            data={
                'domain_name': domain_name,
                'tld': 'ru',
                'preset': 'corp-maillist',
            }
        )

    def add_domain(self, org_id, domain_name):
        return self._make_request(
            self._endpoint_url('proxy/domains/'),
            method='post',
            headers={
                'X-USER-IP': self.ip,
                'X-UID': str(self.admin_uid),
                'X-ORG-ID': str(org_id)
            },
            data={
                'name': domain_name,
            }
        )

    def create_group(self, org_id, login):
        return self._make_request(
            self._endpoint_url('groups/'),
            method='post',
            headers={
                'X-USER-IP': self.ip,
                'X-UID': str(self.admin_uid),
                'X-ORG-ID': str(org_id)
            },
            data={
                'name': login,
                'label': login,
            }
        )

    def create_user(self, org_id, nickname, password, first_name_ru='', first_name_en='', admin_uid=None):
        """Создать пользователя в организации
        https://api-internal.directory.ws.yandex.net/docs/playground.html?version=10#sozdat-novogo-sotrudnika
        """
        user = {
            'nickname': nickname,
            'department_id': 1,
            'name': {
                'first': {
                    'ru': first_name_ru or nickname,
                    'en': first_name_en or nickname,
                },
                'last': {
                    'ru': '',
                    'en': '',
                }
            },
            'birthday': None,
            'gender': None,
            'password': password,
            'password_mode': 'plain',
        }
        headers = {
            'X-USER-IP': self.ip,
            'X-UID': str(admin_uid or self.admin_uid),
            'X-ORG-ID': str(org_id)
        }
        kwargs = dict(
            url=self._endpoint_url('users/'),
            method='post',
            headers=headers,
            data=user,
        )
        return self._make_request(**kwargs)

    def whois(self, email=None, domain=None):
        """
        https://api-internal.directory.ws.yandex.net/docs/playground.html?version=10#informaciya-ob-odnoj-sushnosti-po-domain-ili-email.
        """
        assert (email and not domain) or (not email and domain)
        headers = {
            'X-USER-IP': self.ip,
        }
        params = {}
        if email:
            params['email'] = email
        if domain:
            params['domain'] = domain
        kwargs = dict(
            url=self._endpoint_url('who-is/'),
            method='get',
            params=params,
            headers=headers,
        )
        return self._make_request(**kwargs)

    def list_deputies(self, org_id):
        """
        https://api-internal.directory.ws.yandex.net/docs/playground.html?version=10#poluchit-spisok-vneshnih-zamestitelej-administratorov

        {
            "code": "no-required-scopes",
            "message": "This operation requires one of scopes: {scopes}.",
            "params": {
                "scopes": "directory:read_users"
            }
        }
        """
        headers = {
            'X-USER-IP': self.ip,
            'X-ORG-ID': str(org_id)
        }
        kwargs = dict(
            url=self._endpoint_url('users/deputy/'),
            method='get',
            headers=headers,
        )
        return self._make_request(**kwargs)

    def add_deputy(self, org_id, nickname, admin_uid=None):
        """Нужно выполнять от лица реального админа организации. Можно вызывать несколько раз подряд - идемпотентная"""
        headers = {
            'X-USER-IP': self.ip,
            'X-ORG-ID': str(org_id),
            'X-UID': str(admin_uid or self.admin_uid),
        }
        kwargs = dict(
            url=self._endpoint_url('users/deputy/'),
            method='post',
            headers=headers,
            data={'nickname': nickname, }
        )
        return self._make_request(**kwargs)

    def get_org_info(self, org_id, fields):
        """Requires scope `directory:read_organization`"""
        headers = {
            'X-USER-IP': self.ip,
            # 'X-UID': str(self.admin_uid),
        }
        kwargs = dict(
            url=self._endpoint_url('organizations/{org_id}/'.format(org_id=org_id)),
            method='get',
            params={'fields': ','.join(fields)},
            headers=headers,
        )
        return self._make_request(**kwargs)

    def user_by_uid(self, org_id, user_uid, fields=None):
        headers = {
            'X-USER-IP': self.ip,
            'X-ORG-ID': str(org_id)
        }
        kwargs = dict(
            url=self._endpoint_url('users/{uid}/'.format(uid=user_uid)),
            method='get',
            headers=headers,
            params={'fields': ','.join(fields or [])},
        )
        return self._make_request(**kwargs)


class DirectoryOperationError(DirectoryAPIError):
    """Ошибка операций над DirectoryAPI - будем считать их продолжением в иерархии ошибок DirectoryAPI"""
    pass


class NoOrgWithDomainError(DirectoryOperationError):
    """Нет организации, владеющей доменом"""
    code = 'error_directory_no_org_with_domain'


class NoAccessToOrganization(DirectoryOperationError):
    """yndx.maillists не может создавать группы в организации"""

    code = 'error_directory_no_access_to_organization'

    def __init__(self, message, org_id):
        super(NoAccessToOrganization, self).__init__(message)
        self.org_id = org_id


class AliasIsTaken(DirectoryOperationError):
    """При создании сущности в организации алиас уже занят сущностью другого типа.
    Например, нельзя создать группу так как уже есть пользователь с соответствующим алиасом.
    """
    code = 'error_directory_alias_is_taken'


class NoUid(DirectoryOperationError):
    """Сущность с алиасом существует, но у нее нет UID.
    Например, существует группа с нужным алиасом, но без UID.
    """
    code = 'error_directory_entity_has_no_uid'


class MaillistServiceDisabled(DirectoryOperationError):
    """Сервис maillist не ready и не enabled в организации"""

    code = 'error_directory_maillists_service_disabled_in_organization'

    def __init__(self, message, org_id):
        super(MaillistServiceDisabled, self).__init__(message)
        self.org_id = org_id


class DirectoryMaillistsOperations(object):
    """
    Оберта над DirectoryAPI для адаптации АПИ Директории к нашим сценариям
    Вырыжаем только то что нам нужно в виде вызовов АПИ Директории.
    """

    def __init__(self, uid=settings.DIRECTORY_ADMIN_UID, nickname=settings.DIRECTORY_ADMIN_NICKNAME):
        self._uid = uid
        self._nickname = nickname
        self._directory_api = DirectoryAPI(token=None, admin_uid=uid)

    def get_or_create_organization_with_domain_occupied(self, domain_name, check_org=False):
        """Узнаем организацию которая заняла домен или создаем такую (идемпотентно, кажется)"""
        org_id = None
        try:
            who_is_result = self._directory_api.whois(domain=domain_name)
            org_id = who_is_result['org_id']
        except DirectoryNotFound:
            pass

        if org_id and check_org:
            # проверять доступ до организации имеет смысл только если она уже существует
            if not self.has_admin_role(org_id):
                raise NoAccessToOrganization(message='No admin role in organization', org_id=org_id)
            if not self.is_maillist_service_ready_and_enabled(org_id):
                raise MaillistServiceDisabled(message='Maillists service is disabled in organization', org_id=org_id)

        if org_id:
            return org_id

        # переходим к созданию
        try:
            org_id = self._directory_api.create_org(domain_name)
        except DirectoryAlreadyExistsError as exc:
            # уже есть организация в которой maillists является админом и в которую добавлен домен
            org_id = exc.response_data['params']['conflicting_org_id']

        # доп проверка семантики АПИ: ожидается, что admin_uid является админом в этой точке кода
        org_info = self._directory_api.get_org_info(org_id=org_id, fields=('admin_id', 'services', 'domains'))
        assert self._uid == org_info['admin_id']

        # отмечаем домен, как подтвержденный
        try:
            self._directory_api.add_domain(org_id=org_id, domain_name=domain_name)
        except DirectoryAlreadyExistsError:
            org_info = self._directory_api.get_org_info(org_id=org_id, fields=('admin_id', 'domains'))
            assert self._uid == org_info['admin_id'] and domain_name in set(org_info['domains']['owned'])

        return org_id

    def has_admin_role(self, org_id):
        """По заданному org_id проверяем может ли UID создавать юзеров или группы
        Для нас это значит что UID либо админ, либо зам. админа.
        """

        org_info = self._directory_api.get_org_info(org_id, fields=('admin_id',))
        if self._uid == org_info['admin_id']:
            return True

        try:
            user_info = self._directory_api.user_by_uid(
                org_id=org_id,
                user_uid=self._uid,
                fields=['nickname', 'id', 'role', 'user_type', 'is_admin', 'is_robot'],
            )
            # предполагаем, что если maillists является внутренним пользователем организации, то он ее создал,
            # т. е. он там админ, а если вдруг оказалось, что yndx-maillists не админ - это очень странно, паникуем
            assert user_info['is_admin']
        except DirectoryNotFound:
            pass

        # на текущий момент заместители админов НЕ принадлежат организации,
        # я слышал, что это поведение может измениться?
        deputy_nicknames = set(self._directory_api.list_deputies(org_id)['deputies'])
        return settings.DIRECTORY_ADMIN_NICKNAME in deputy_nicknames

    def is_maillist_service_ready_and_enabled(self, org_id):
        """Ответ имеет смысл только для организаций с хотя бы одним подтвержденным доменом"""

        org_info = self._directory_api.get_org_info(org_id=org_id, fields=('services',))

        # проверям статус сервиса maillist (интересно, что maillist относится ко внешнему сервису рассылок)
        # сервис должен автоматически быть ready и enabled после подтверждения домена,
        # а если не так, то нужно призывать дежурных Конекта за помощью
        for service in org_info['services']:
            if service['slug'] == settings.DIRECTORY_MAILLIST_SERVICE_SLUG:
                if service['ready'] and service['enabled']:
                    return True
                break
        return False

    def create_group_in_organization_by_domain(self, domain_name, group_alias):
        try:
            org_id = self._directory_api.whois(domain=domain_name)['org_id']
        except DirectoryNotFound:
            raise NoOrgWithDomainError(
                message='No organization found for domain {domain}'.format(domain=domain_name))

        if not self.has_admin_role(org_id):
            raise NoAccessToOrganization(
                message='Unable to create users in organization {org_id}'.format(org_id=org_id))

        try:
            group_uid = self._directory_api.create_group(org_id, group_alias)['uid']
        except DirectoryAlreadyExistsError:
            desired_email = '{group_alias}@{domain_name}'.format(group_alias=group_alias, domain_name=domain_name)
            obj = self._directory_api.whois(email=desired_email)
            if obj['type'] != 'group':
                raise AliasIsTaken(
                    message='Entity {email} of type {type} already exists in organization {org_id}'.format(
                        email=desired_email, type=obj['type'], org_id=obj['org_id'])
                )

            group_id = obj['object_id']
            group_uid = self._directory_api.get_group_info(org_id, group_id, fields=['id', 'uid'])['uid']

            if not group_uid:
                raise NoUid(
                    message='Group id={group_id} alias={alias} exists without UID'.format(
                        group_id=group_id, alias=group_alias)
                )

        return group_uid

    def get_user_in_domain(self, domain_name, nickname):
        """Получить пользователя nickname@domain"""
        email = '{nickname}@{domain}'.format(nickname=nickname, domain=domain_name)
        obj = self._directory_api.whois(email=email)
        if obj['type'] != 'user':
            raise AliasIsTaken(message='Alias is taken by object of not user type')
        return self._directory_api.user_by_uid(org_id=obj['org_id'], user_uid=obj['object_id'],
                                               fields=self._directory_api.USER_ALL_FIELDS)

    def create_user_for_domain(self, domain_name, nickname, password, first_name_en='', first_name_ru=''):
        """Создать обычного пользователя в нужном домене (попутно ищем или создаем организацию)"""
        org_id = self.get_or_create_organization_with_domain_occupied(domain_name=domain_name, check_org=True)
        user_data = self._directory_api.create_user(
            org_id, nickname=nickname, password=password, first_name_en=first_name_en, first_name_ru=first_name_ru)
        return user_data

    def _fix_access_to_org(self, org_id):
        """
        Исправляем доступ yndx-maillists к организации в коннекте.
        Фикс на данный момент один - добавить юзера в депьюти.
        ВАЖНО: это не следует делать при обработке запросов из UI, данная операция предназначена только для ручного
        использования в manage.py команде при ручном разборе инцидентов.
        """
        if self.has_admin_role(org_id):
            return
        org_info = self._directory_api.get_org_info(org_id=org_id, fields=('admin_id',))
        org_admin_uid = org_info['admin_id']
        # "от имени админа организации нарекаю yndx-maillists зам админа"
        self._directory_api.add_deputy(org_id, nickname=settings.DIRECTORY_ADMIN_NICKNAME, admin_uid=org_admin_uid)
