# encoding: utf-8
import logging
from collections import deque
from django.conf import settings


from mlcore.ml.models import MailList, User
from mlcore.subscribe import subscription_type
from .exceptions import DirectoryAPIError, RetryProcess
from .api import DirectoryAPI

logger = logging.getLogger(__name__)


class BaseEventProcessor(object):
    event_name = None

    def __init__(self, data):
        self.data = data
        self.api = DirectoryAPI()

    def process(self):
        raise NotImplementedError


class GroupAddedEvent(BaseEventProcessor):
    event_name = 'group_added'

    def _generate_membership_event_data(self):
        users_data = [user for user in self.data['object']['members'] if user['type'] == 'user']
        group_id = self.data['object']['id']

        return {
            'content': {
                'diff': {
                    'members': {
                        'add': {'users': [user['object']['id'] for user in users_data], 'groups': []},
                        'remove': {'users': [], 'groups': []}
                    }
                }
            },
            'object': {
                'id': group_id
            },
            'event': GroupMembershipChanged.event_name
        }

    def process(self):
        from mlcore.tasks import operations
        if self.data['object']['email']:
            operations.create_maillist.delay(
                {
                    'initiator': 'directory_notification',
                    'comment': u'Создание рассылки по группе Директории'
                },
                data={
                    'name': self.data['object']['email'],
                    'responsible': [user['id'] for user in self.data['object']['admins']],
                    'info': self.data['object']['description'].get('ru', ''),
                    'info_en': self.data['object']['description'].get('en', ''),
                    'is_open': False,
                    'is_internal': True
                },
                use_cmail=False
            )

            from mlcore.tasks.directory import process_notification
            # Т.к.
            membership_changed_event = self._generate_membership_event_data()
            process_notification.delay(membership_changed_event)


class GroupDeletedEvent(BaseEventProcessor):
    event_name = 'group_deleted'

    def process(self):
        try:
            maillist = MailList.objects.get(email=self.data['object']['email'])
            maillist.is_deleted = True
            maillist.save()

            maillist.set_as_modified()
        except MailList.DoesNotExist:
            logger.error('Mail list not found: %s', self.data['object']['email'])


class GroupMembershipChanged(BaseEventProcessor):
    event_name = 'group_membership_changed'

    @staticmethod
    def __subscribe_users(maillist, uids):
        from mlcore.tasks import operations
        # Подписываем всегда в INBOX
        stype = subscription_type.INBOX
        user_list = User.objects.filter(staff__uid__in=uids)

        for user in user_list:
            if user.subscriptions.filter(list=maillist, is_sub=True).exists():
                continue

            operations.subscribe_user.delay({
                'initiator': 'directory_notification',
                'comment': u'Подписка при добавлении в группу Директории'
            },
                user, maillist.name, stype, check_rights=False
            )

    @staticmethod
    def __unsubscribe_users(maillist, uids):
        from mlcore.tasks import base
        user_list = User.objects.filter(staff__uid__in=uids)

        for user in user_list:
            if not user.subscriptions.filter(list=maillist, is_sub=True).exists():
                continue

            base.unsubscribe_inbox({
                'initiator': 'directory_notification',
                'comment': u'Отписка при удалении из группы Директории'
            }, user, maillist.name).delay()

    @staticmethod
    def __subscribe_groups(maillist, group_emails):
        from mlcore.tasks import operations
        for email in group_emails:
            operations.subscribe_email({
                'initiator': 'directory_notification',
                'comment': u'Подписка при добавлении группы в другую группу Директории'
            }, email, maillist.name).delay()

    @staticmethod
    def __unsubscribe_groups(maillist, group_emails):
        from mlcore.tasks import base
        for email in group_emails:
            base.unsubscribe_inbox({
                'initiator': 'directory_notification',
                'comment': u'Отписка при удалении группы из группы Директории'
            }, email, maillist.name).delay()

    def __get_group_emails_to_subscribe(self, group_id):
        """
        Обход в ширину предков группы для поиска email'ов, на которые нужно подписать
        """
        queue = deque()
        emails = set()

        try:
            group_info = self.api.get_group_info(settings.DIRECTORY_ORG_ID, group_id)
            queue.append(group_info)

            while queue:
                group = queue.popleft()
                email = group['email']
                if email:
                    emails.add(email)
                else:
                    for parent_group in group['member_of']:
                        queue.append(self.api.get_group_info(settings.DIRECTORY_ORG_ID, parent_group['id']))
        except DirectoryAPIError as e:
            logger.error('Directory api error: %s', e.message)

        return list(emails)

    def __get_group_uids_emails(self, group_id):
        """
        Обход в ширину потомков группы, чтобы получить всех пользователей и email'ы групп
        """
        queue = deque()

        uids = set()
        emails = set()

        try:
            group_info = self.api.get_group_info(settings.DIRECTORY_ORG_ID, group_id)
            queue.append(group_info)

            while queue:
                group = queue.popleft()
                email = group['email']

                if email:
                    emails.add(email)
                else:
                    for member in group['members']:
                        if member['type'] == 'user':
                            uids.add(member['object']['id'])
                        elif member['type'] == 'group':
                            queue.append(self.api.get_group_info(settings.DIRECTORY_ORG_ID, member['object']['id']))
        except DirectoryAPIError as e:
            logger.error('Directory api error: %s', e.message)

        return uids, emails

    def process(self):
        """
        Подписываем и отписываем пользователей и другие группы
        """

        content = self.data['content']

        subscribe_users = set()
        unsubscribe_users = set()

        subscribe_emails = set()
        unsubscribe_emails = set()

        for uid in content['diff']['members']['add']['users']:
            subscribe_users.add(uid)

        for uid in content['diff']['members']['remove']['users']:
            unsubscribe_users.add(uid)

        for group_id in content['diff']['members']['add']['groups']:
            group_subscribe_users, group_subscribe_emails = self.__get_group_uids_emails(group_id)
            subscribe_users |= group_subscribe_users
            subscribe_emails |= group_subscribe_emails

        for group_id in content['diff']['members']['remove']['groups']:
            group_subscribe_users, group_subscribe_emails = self.__get_group_uids_emails(group_id)
            unsubscribe_users |= group_subscribe_users
            unsubscribe_emails |= group_subscribe_emails

        for email in self.__get_group_emails_to_subscribe(self.data['object']['id']):
            try:
                maillist = MailList.objects.get(email=email)

                self.__subscribe_users(maillist, subscribe_users)
                self.__unsubscribe_users(maillist, unsubscribe_users)

                self.__subscribe_groups(maillist, subscribe_emails)
                self.__unsubscribe_groups(maillist, unsubscribe_emails)
            except MailList.DoesNotExist:
                # Тут что-то надо сделать с тем, что может быть вызвано событие до того, как рассылка создалась
                logger.error('Mail list not found: %s', email)
                raise RetryProcess
