# encoding: utf-8
import logging
from django.conf import settings

from .exceptions import EventNotSupported, ParseError
from .events import GroupAddedEvent, GroupDeletedEvent, GroupMembershipChanged

logger = logging.getLogger(__name__)


class EventProcessor(object):
    EVENTS = (
        GroupAddedEvent,
        GroupDeletedEvent,
        GroupMembershipChanged
    )

    def __init__(self):
        self.__events_cls = {}
        for cls in self.EVENTS:
            self.__events_cls[cls.event_name] = cls

    @staticmethod
    def _parse_event_name(data):
        return data.get('event')

    @staticmethod
    def _parse_org_id(data):
        return data.get('org_id')

    def can_process(self, data):
        return (self._parse_event_name(data) in self.__events_cls
                and self._parse_org_id(data) == settings.DIRECTORY_ORG_ID)

    def process(self, data):
        event_cls = self.__events_cls.get(self._parse_event_name(data))

        if not event_cls:
            raise EventNotSupported

        logger.info('Start %s event processing', event_cls.event_name)
        try:
            event_cls(data).process()
        except KeyError as e:
            raise ParseError('message parse error: %s', e.message)


processor = EventProcessor()

