import os.path

from library.python import resource

from mail.devpack.lib import helpers
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.devpack.lib.components.base import BaseComponent
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvmapi import TvmApi


class MdbSaveBase(BaseComponent):
    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(MdbSaveBase, cls).gen_config(port_generator, config=config)
        http = YplatformHttpService.gen_config(port_generator, config=config)
        return {
            **base,
            **http,
        }

    def __init__(self, env, components):
        super().__init__(env, components)
        self.yhttp = YplatformHttpService(env, self.NAME, binary_name='mdbsave', custom_path='mdbsave')

    def init_root(self):
        self.yhttp.init_root()

        helpers.mkdir_recursive(self.app_config_path)

        self.write_resource('macs_pg/query.conf', os.path.join(self.app_config_path, 'query.conf'))
        self.write_resource('mdbsave/base.yml', os.path.join(self.app_config_path, 'base.yml'))
        self.write_resource('mdbsave/tvm_secret', os.path.join(self.secrets_path, 'tvm_secret'))
        self.write_resource('mdbsave/dev-cert.pem', os.path.join(self.secrets_path, 'dev-cert.pem'))
        self.write_resource('mdbsave/dev-key.pem', os.path.join(self.secrets_path, 'dev-key.pem'))

        self._init_root()

    def _init_root(self):
        pass

    def write_resource(self, resource_path, destination_path):
        helpers.write2file(what=resource.find(resource_path), path=destination_path)

    @property
    def etc_path(self):
        return self.yhttp.get_etc_path()

    @property
    def app_path(self):
        return self.yhttp.get_app_path()

    @property
    def app_config_path(self):
        return os.path.join(self.app_path, 'config')

    @property
    def secrets_path(self):
        return self.yhttp.get_secrets_path()

    def get_root(self):
        return self.yhttp.get_root()

    def webserver_port(self):
        return self.yhttp.webserver_port

    def start(self):
        self.yhttp.start('pong')

    def stop(self):
        self.yhttp.stop()

    def restart(self):
        self.stop()
        self.start()

    def info(self):
        info = self.yhttp.info()
        return {
            **info,
            'state': self.state
        }

    def ping(self):
        return self.yhttp.ping()


class MdbSaveComponent(MdbSaveBase):
    NAME = 'mdbsave'
    DEPS = [Sharpei, TvmApi]

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(MdbSaveComponent, cls).gen_config(port_generator, config=config)
        return {
            **base,
        }

    def make_config(self, params={}):
        default_params = dict(
            tvm_api_port=self.components[TvmApi].port,
            sharpei_port=self.components[Sharpei].webserver_port(),
        )
        params = {**default_params, **params}
        return self.yhttp.format_config(
            config=resource.find('mdbsave/devpack.yml'),
            **params
        )

    def __init__(self, env, components):
        super().__init__(env, components)

    def _init_root(self):
        devpack_config = self.make_config()
        helpers.write2file(what=devpack_config, path=self.yhttp.get_config_path())


class MdbSaveTesting(MdbSaveBase):
    NAME = 'mdbsave_testing'
    DEPS = []

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(MdbSaveTesting, cls).gen_config(port_generator, config=config)
        return base

    def __init__(self, env, components):
        super().__init__(env, components)

    def _init_root(self):
        devpack_config = self.yhttp.format_config(config=resource.find('mdbsave/testing.yml'))
        helpers.write2file(what=devpack_config, path=self.yhttp.get_config_path())
