#include "build_envelope.h"

#include <mail/library/utf8/utf8.h>
#include <butil/email/helpers.h>

namespace NMdb {

void SetAttaches(macs::EnvelopeFactory& factory, const std::vector<TAttachment>& attachments) {
    for (const auto& attachment: attachments) {
        auto name = NUtil::Utf8ByteHead(attachment.Name, 250);
        auto type = NUtil::Utf8ByteHead(attachment.Type, 250);
        auto hid  = NUtil::Utf8ByteHead(attachment.Hid, 250);

        factory.addAttachment(macs::AttachmentDescriptor(hid, type, name, attachment.Size));
    }
}

void SetHeaders(macs::EnvelopeFactory& factory, const THeaders& headers, TContextPtr /*taskCtx*/) {
    // Subject
    factory.subject(NUtil::Utf8ByteHead(headers.Subject.empty() ? "No subject" : headers.Subject, 767));

    // Message-ID:
    factory.rfcId(NUtil::Utf8ByteHead(headers.MessageId, 1023));

    /// Date:
    factory.date(headers.Date ? headers.Date : time(nullptr));

    /// To:
    const auto& to = headers.To.empty() ? "No address" : headers.To;
    factory.to(to);
    // parse 'To' header here for display names size postprocessing
    macs::Emails emails = EmailHelpers::toEmailVec(to);
    for (auto& email : emails) {
        constexpr const size_t displayNameMaxLen = 1024;
        if (email.displayName().size() > displayNameMaxLen) {
            std::string displayNameTruncated = NUtil::Utf8ByteHead(email.displayName(), displayNameMaxLen);
            email.setDisplayName(std::move(displayNameTruncated));
        }
    }
    factory.parsedTo(std::move(emails));

    /// From:
    factory.from(NUtil::Utf8ByteHead(headers.From.empty() ? "No address" : headers.From, 1023));

    /// Reply-To:
    factory.replyTo(NUtil::Utf8ByteHead(headers.ReplyTo, 2047));

    /// To: + CC:
    factory.extraData(NUtil::Utf8ByteHead(headers.ToAndCc, 1023));

    /// CC:
    factory.cc(NUtil::Utf8ByteHead(headers.Cc, 2047));

    /// BCC:
    factory.bcc(NUtil::Utf8ByteHead(headers.Bcc, 2047));

    /// In-Reply-To:
    factory.inReplyTo(NUtil::Utf8ByteHead(headers.InReplyTo, 2047));
}

macs::Envelope BuildEnvelope(
    const TMessageBase& message,
    const std::string& fid,
    const std::vector<TLid>& lids,
    const std::optional<macs::Tab::Type>& tab,
    const THeaders& headers,
    const std::vector<TAttachment>& attachments,
    TContextPtr taskCtx)
{
    macs::EnvelopeFactory factory;

    factory.fid(fid);

    factory.stid(message.Stid);

    factory.firstline(NUtil::Utf8ByteHead(message.FirstLine, 384));

    factory.uidl(message.Pop3Uidl);

    factory.size(message.Size);

    factory.receiveDate(message.ReceivedDate ? message.ReceivedDate : time(nullptr));

    SetHeaders(factory, headers, taskCtx);

    SetAttaches(factory, attachments);

    factory.addLabelIDs(lids);

    factory.tab(tab);

    return factory.release();
}

macs::MimeParts BuildMimeParts(const std::vector<TMimePart>& parts) {
    macs::MimeParts ret;
    for (const auto& part: parts) {
        macs::MimePartFactory factory;
        factory
            .hid(part.Hid)
            .contentType(part.ContentType)
            .contentSubtype(part.ContentSubtype)
            .boundary(part.Boundary)
            .name(part.Name)
            .charset(part.Charset)
            .encoding(part.Encoding)
            .contentDisposition(part.ContentDisposition)
            .fileName(part.FileName)
            .cid(part.ContentId)
            .offsetBegin(part.Offset)
            .offsetEnd(part.Offset + part.Length);
        ret.emplace_back(factory.release());
    }
    return ret;
}

macs::ThreadsMergeRules MergeRule(EThreadsMergeRules rule) {
    switch (rule) {
        case EThreadsMergeRules::Hash:
            return macs::ThreadsMergeRules::hash;
        case EThreadsMergeRules::References:
            return macs::ThreadsMergeRules::references;
        case EThreadsMergeRules::ForceNewThread:
            return macs::ThreadsMergeRules::forceNewThread;
    }
}

macs::ThreadsHashNamespaces HashNamespace(EThreadsHashNamespaces ns) {
    switch (ns) {
        case EThreadsHashNamespaces::Subject:
            return macs::ThreadsHashNamespaces::subject;
        case EThreadsHashNamespaces::From:
            return macs::ThreadsHashNamespaces::from;
    }
};

macs::ThreadMeta BuildThreadMeta(const TThreadInfo& info) {
    macs::ThreadMeta ret;
    ret.hash.ns = HashNamespace(info.Hash.Ns);
    ret.hash.key = info.Hash.Key;
    ret.hash.value = info.Hash.Value;
    ret.limits.receivedDate = info.Limits.ReceivedDate;
    ret.limits.daysLimit = info.Limits.DaysLimit;
    ret.limits.countLimit = info.Limits.CountLimit;
    ret.mergeRule = MergeRule(info.MergeRule);
    ret.referenceHashes = info.ReferenceHashes;
    ret.messageIds = info.MessageIds;
    ret.inReplyToHash = info.InReplyToHash;
    ret.messageIdHash = info.MessageIdHash;
    return ret;
}

} // namespace NMdb
